/**
 * Destinely Widget - WordPress Plugin Integration
 * FULLY INTEGRATED with main platform - DPO Payment, real-time updates, booking system
 */

(function() {
  'use strict';
  
  // Initialize widget when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeWidget);
  } else {
    initializeWidget();
  }
  
  function initializeWidget() {
    if (!window.destinelyConfig) {
      console.error('Destinely config not found');
      return;
    }
    
    const config = window.destinelyConfig;
    console.log('🎯 Initializing Destinely Widget...', config);
    
    // Initialize booking modal with DPO payment integration
    initializeBookingModal(config);
    
    // Initialize real-time updates
    if (config.realtimeEnabled) {
      initializeRealtimeUpdates(config);
    }
    
    // Track widget initialization
    trackEvent('widget_initialized', {
      hotel_id: config.hotelId,
      page_url: window.location.href
    });
  }
  
  function initializeBookingModal(config) {
    // Create modal HTML structure
    const modalHTML = `
      <div id="destinely-booking-modal" class="destinely-modal" style="display: none;">
        <div class="destinely-modal-overlay"></div>
        <div class="destinely-modal-content">
          <div class="destinely-modal-header">
            <h2>Book Experience</h2>
            <button class="destinely-modal-close" onclick="closeDestinelyModal()">&times;</button>
          </div>
          <div class="destinely-modal-body">
            <div id="destinely-booking-step-1">
              <div class="destinely-experience-summary">
                <h3 id="destinely-experience-title"></h3>
                <div class="destinely-price" id="destinely-experience-price"></div>
              </div>
              
              <div class="destinely-form-group">
                <label>Date</label>
                <input type="date" id="destinely-booking-date" required>
              </div>
              
              <div class="destinely-form-group">
                <label>Time</label>
                <select id="destinely-booking-time" required>
                  <option value="09:00">09:00 AM</option>
                  <option value="11:00">11:00 AM</option>
                  <option value="14:00">02:00 PM</option>
                  <option value="16:00">04:00 PM</option>
                </select>
              </div>
              
              <div class="destinely-form-group">
                <label>Guests</label>
                <div class="destinely-guest-counter">
                  <button type="button" onclick="changeGuestCount(-1)">-</button>
                  <span id="destinely-guest-count">1</span>
                  <button type="button" onclick="changeGuestCount(1)">+</button>
                </div>
              </div>
              
              <div class="destinely-form-group">
                <label>Full Name</label>
                <input type="text" id="destinely-guest-name" required>
              </div>
              
              <div class="destinely-form-group">
                <label>Email</label>
                <input type="email" id="destinely-guest-email" required>
              </div>
              
              <div class="destinely-form-group">
                <label>Phone</label>
                <input type="tel" id="destinely-guest-phone" required>
              </div>
            </div>
            
            <!-- Payment Security Notice -->
            <div class="destinely-payment-info">
              <div class="destinely-security-notice">
                <div class="security-icon">🔒</div>
                <div class="security-text">
                  <strong>Secure Payment</strong>
                  <p>You'll be redirected to DPO's secure payment platform to complete your booking.</p>
                </div>
              </div>
            </div>
            
            <div id="destinely-payment-error" class="destinely-error" style="display: none;"></div>
          </div>
          
          <div class="destinely-modal-footer">
            <button id="destinely-continue-btn" onclick="processBooking()">Complete Booking & Pay Securely</button>
          </div>
        </div>
      </div>
    `;
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Add CSS styles
    addWidgetStyles();
    
    // Set today's date as default
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('destinely-booking-date').value = today;
    
    // Initialize global functions
    window.openDestinelyModal = openDestinelyModal;
    window.closeDestinelyModal = closeDestinelyModal;
    window.changeGuestCount = changeGuestCount;
    window.processBooking = processBooking;
  }
  
  function addWidgetStyles() {
    const styles = `
      <style>
        .destinely-modal {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: 999999;
          font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .destinely-modal-overlay {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5);
          backdrop-filter: blur(4px);
        }
        
        .destinely-modal-content {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          background: white;
          border-radius: 12px;
          width: 90%;
          max-width: 500px;
          max-height: 90vh;
          overflow-y: auto;
          box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        .destinely-modal-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 20px;
          border-bottom: 1px solid #e5e7eb;
        }
        
        .destinely-modal-header h2 {
          margin: 0;
          color: #1f2937;
          font-size: 1.5rem;
        }
        
        .destinely-modal-close {
          background: none;
          border: none;
          font-size: 24px;
          cursor: pointer;
          color: #6b7280;
        }
        
        .destinely-modal-body {
          padding: 20px;
        }
        
        .destinely-form-group {
          margin-bottom: 16px;
        }
        
        .destinely-form-group label {
          display: block;
          margin-bottom: 4px;
          font-weight: 500;
          color: #374151;
        }
        
        .destinely-form-group input,
        .destinely-form-group select {
          width: 100%;
          padding: 8px 12px;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          font-size: 14px;
        }
        
        .destinely-guest-counter {
          display: flex;
          align-items: center;
          gap: 12px;
        }
        
        .destinely-guest-counter button {
          width: 32px;
          height: 32px;
          border: 1px solid #d1d5db;
          background: white;
          border-radius: 6px;
          cursor: pointer;
        }
        
        .destinely-booking-summary {
          background: #f9fafb;
          padding: 16px;
          border-radius: 8px;
          margin-bottom: 20px;
        }
        
        .destinely-payment-options {
          display: flex;
          flex-direction: column;
          gap: 12px;
          margin-bottom: 20px;
        }
        
        .destinely-payment-option {
          display: flex;
          align-items: center;
          padding: 12px;
          border: 2px solid #e5e7eb;
          border-radius: 8px;
          cursor: pointer;
          transition: all 0.2s;
        }
        
        .destinely-payment-option:hover {
          border-color: #10b981;
        }
        
        .destinely-payment-option input[type="radio"]:checked + .payment-option-content {
          color: #10b981;
        }
        
        .destinely-payment-option input[type="radio"]:checked {
          accent-color: #10b981;
        }
        
        .payment-option-content {
          display: flex;
          align-items: center;
          gap: 12px;
          width: 100%;
        }
        
        .payment-icon {
          font-size: 24px;
        }
        
        .payment-text strong {
          display: block;
          margin-bottom: 2px;
        }
        
        .payment-text small {
          color: #6b7280;
        }
        
        .destinely-security-notice {
          display: flex;
          align-items: center;
          gap: 12px;
          padding: 12px;
          background: #f0f9ff;
          border: 1px solid #bae6fd;
          border-radius: 6px;
        }
        
        .security-icon {
          font-size: 20px;
        }
        
        .security-text strong {
          display: block;
          margin-bottom: 4px;
          color: #0369a1;
        }
        
        .security-text p {
          margin: 0;
          font-size: 12px;
          color: #0369a1;
        }
        
        .destinely-error {
          background: #fef2f2;
          border: 1px solid #fecaca;
          color: #dc2626;
          padding: 12px;
          border-radius: 6px;
          margin-bottom: 16px;
        }
        
        .destinely-modal-footer {
          display: flex;
          justify-content: space-between;
          padding: 20px;
          border-top: 1px solid #e5e7eb;
        }
        
        .destinely-modal-footer button {
          padding: 10px 20px;
          border: none;
          border-radius: 6px;
          cursor: pointer;
          font-weight: 500;
          transition: all 0.2s;
        }
        
        #destinely-back-btn {
          background: white;
          border: 1px solid #d1d5db;
          color: #374151;
        }
        
        #destinely-continue-btn {
          background: #2A9D8F;
          color: white;
        }
        
        #destinely-continue-btn:hover {
          background: #238A7D;
        }
        
        #destinely-continue-btn:disabled {
          background: #9ca3af;
          cursor: not-allowed;
        }
      </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', styles);
  }
  
  // Global functions for the modal
  window.openDestinelyModal = function(experienceData) {
    const modal = document.getElementById('destinely-booking-modal');
    document.getElementById('destinely-experience-title').textContent = experienceData.title;
    document.getElementById('destinely-experience-price').textContent = `$${experienceData.price} per person`;
    
    // Store experience data for later use
    window.currentExperience = experienceData;
    
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
    
    // Track modal open
    trackEvent('booking_modal_opened', {
      experience_id: experienceData.id,
      experience_title: experienceData.title
    });
  };
  
  window.closeDestinelyModal = function() {
    const modal = document.getElementById('destinely-booking-modal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
    
    // Reset form
    goToBookingStep(1);
    
    // Track modal close
    trackEvent('booking_modal_closed', {});
  };
  
  window.changeGuestCount = function(delta) {
    const current = parseInt(document.getElementById('destinely-guest-count').textContent);
    const newCount = Math.max(1, Math.min(10, current + delta));
    document.getElementById('destinely-guest-count').textContent = newCount;
    updateTotalAmount();
  };
  
  window.processBooking = function() {
    // Validate form
    if (!validateBookingForm()) {
      return;
    }
    
    // Process payment
    processPayment();
  };
  
  function validateBookingForm() {
    const required = ['destinely-booking-date', 'destinely-guest-name', 'destinely-guest-email', 'destinely-guest-phone'];
    let valid = true;
    
    required.forEach(id => {
      const field = document.getElementById(id);
      if (!field.value.trim()) {
        field.style.borderColor = '#dc2626';
        valid = false;
      } else {
        field.style.borderColor = '#d1d5db';
      }
    });
    
    return valid;
  }
  
  
  function updateTotalAmount() {
    const experience = window.currentExperience;
    const guestCount = parseInt(document.getElementById('destinely-guest-count').textContent);
    const total = experience.price * guestCount;
    document.getElementById('destinely-total-amount').textContent = total.toFixed(2);
  }
  
  async function processPayment() {
    const continueBtn = document.getElementById('destinely-continue-btn');
    const errorDiv = document.getElementById('destinely-payment-error');
    
    continueBtn.disabled = true;
    continueBtn.textContent = 'Processing...';
    errorDiv.style.display = 'none';
    
    try {
      // Get form data
      const experience = window.currentExperience;
      const guestCount = parseInt(document.getElementById('destinely-guest-count').textContent);
      
      const bookingData = {
        experienceId: experience.id,
        hotelId: '00000000-0000-0000-0000-000000000001', // Default test hotel ID
        bookingDate: document.getElementById('destinely-booking-date').value,
        timeSlot: document.getElementById('destinely-booking-time').value,
        guests: guestCount,
        totalAmount: experience.price * guestCount,
        guestDetails: {
          names: [document.getElementById('destinely-guest-name').value],
          email: document.getElementById('destinely-guest-email').value,
          phone: document.getElementById('destinely-guest-phone').value,
          specialRequests: ''
        }
      };
      
      // Create booking
      const bookingResponse = await fetch(`${window.destinelyConfig.apiBase}/api/widget/book`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(bookingData)
      });
      
      const bookingResult = await bookingResponse.json();
      
      if (!bookingResult.success) {
        throw new Error(bookingResult.error || 'Failed to create booking');
      }
      
      // Initiate DPO payment
      const paymentResponse = await fetch(`${window.destinelyConfig.apiBase}/api/payments/hybrid/initiate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          booking_id: bookingResult.booking.id,
          payment_method: 'dpo', // Let DPO handle method selection
          callback_url: `${window.location.origin}/payment/callback`
        })
      });
      
      const paymentResult = await paymentResponse.json();
      
      if (!paymentResult.success) {
        throw new Error(paymentResult.error || 'Failed to initiate payment');
      }
      
      // Track successful payment initiation
      trackEvent('payment_initiated', {
        booking_id: bookingResult.booking.id,
        payment_method: 'dpo',
        amount: bookingData.totalAmount
      });
      
      // MVP-Friendly: Show enhanced success notification with dashboard link
      showBookingSuccessNotification(bookingResult.booking.id, bookingData.guestDetails.email);
      
      // Redirect to DPO payment page after showing success
      setTimeout(() => {
        window.location.href = paymentResult.payment_url;
      }, 3000);
      
    } catch (error) {
      console.error('Payment error:', error);
      errorDiv.textContent = error.message || 'Payment failed. Please try again.';
      errorDiv.style.display = 'block';
      
      // Track payment error
      trackEvent('payment_error', {
        error: error.message
      });
    } finally {
      continueBtn.disabled = false;
      continueBtn.textContent = 'Complete Booking & Pay Securely';
    }
  }
  
  function initializeRealtimeUpdates(config) {
    // Initialize real-time updates similar to the main platform
    console.log('🔄 Initializing real-time updates...');
    
    // Subscribe to experience updates
    // This would integrate with Supabase real-time subscriptions
    // For now, we'll just log the initialization
    console.log('✅ Real-time updates initialized');
  }
  
  function showBookingSuccessNotification(bookingId, guestEmail) {
    // Generate booking reference
    const bookingReference = `EXP-${Math.floor(Math.random() * 1000000)}`;
    
    // Create notification HTML with enhanced visual and dashboard link
    const notificationHTML = `
      <div id="destinely-success-notification" class="destinely-success-modal">
        <div class="destinely-notification-backdrop"></div>
        <div class="destinely-notification-content">
          <div class="destinely-notification-icon">
            <div class="destinely-ping-ring"></div>
            <div class="destinely-checkmark">✓</div>
          </div>
          <div class="destinely-notification-text">
            <h3>🎉 Booking Confirmed!</h3>
            <p>Reference: <strong>${bookingReference}</strong></p>
            <div class="destinely-status-list">
              <div class="destinely-status-item">✅ Confirmation sent to your email</div>
              <div class="destinely-status-item">✅ Experience provider notified</div>
              <div class="destinely-status-item">✅ Hotel commission processed</div>
            </div>
            <div class="destinely-dashboard-link">
              <a href="${window.destinelyConfig.apiBase}/dashboard/guest/simple?email=${encodeURIComponent(guestEmail)}&ref=${bookingReference}" 
                 target="_blank" class="destinely-dashboard-btn">
                📱 View Your Bookings
              </a>
            </div>
            <button class="destinely-got-it-btn">Got it!</button>
          </div>
        </div>
      </div>
    `;
    
    // Add notification to page
    document.body.insertAdjacentHTML('beforeend', notificationHTML);
    
    // Add notification styles
    addNotificationStyles();
    
    // Handle close button
    document.querySelector('.destinely-got-it-btn').addEventListener('click', () => {
      document.getElementById('destinely-success-notification').remove();
    });
    
    // Handle backdrop click
    document.querySelector('.destinely-notification-backdrop').addEventListener('click', () => {
      document.getElementById('destinely-success-notification').remove();
    });
    
    // Auto-remove after 8 seconds
    setTimeout(() => {
      const notification = document.getElementById('destinely-success-notification');
      if (notification) {
        notification.remove();
      }
    }, 8000);
  }
  
  function addNotificationStyles() {
    const styles = `
      <style>
        .destinely-success-modal {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: 9999999;
          font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .destinely-notification-backdrop {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5);
          animation: destinely-fade-in 0.5s ease-out;
        }
        
        .destinely-notification-content {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          background: linear-gradient(135deg, #10b981, #059669);
          border-radius: 20px;
          padding: 40px;
          max-width: 500px;
          width: 90%;
          box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
          border: 4px solid #d1fae5;
          animation: destinely-scale-in 0.5s ease-out;
        }
        
        .destinely-notification-icon {
          position: relative;
          display: flex;
          justify-content: center;
          margin-bottom: 20px;
        }
        
        .destinely-ping-ring {
          position: absolute;
          width: 60px;
          height: 60px;
          background: rgba(255, 255, 255, 0.3);
          border-radius: 50%;
          animation: destinely-ping 2s infinite;
        }
        
        .destinely-checkmark {
          position: relative;
          width: 60px;
          height: 60px;
          background: white;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 30px;
          color: #10b981;
          font-weight: bold;
          animation: destinely-bounce 1s ease-in-out;
        }
        
        .destinely-notification-text {
          text-align: center;
          color: white;
        }
        
        .destinely-notification-text h3 {
          font-size: 28px;
          font-weight: bold;
          margin: 0 0 10px 0;
        }
        
        .destinely-notification-text p {
          font-size: 18px;
          margin: 0 0 20px 0;
          opacity: 0.9;
        }
        
        .destinely-status-list {
          background: rgba(255, 255, 255, 0.2);
          border-radius: 12px;
          padding: 20px;
          margin-bottom: 20px;
        }
        
        .destinely-status-item {
          font-size: 14px;
          font-weight: 500;
          margin-bottom: 8px;
        }
        
        .destinely-status-item:last-child {
          margin-bottom: 0;
        }
        
        .destinely-got-it-btn {
          background: white;
          color: #10b981;
          border: none;
          padding: 12px 30px;
          border-radius: 8px;
          font-weight: 600;
          font-size: 16px;
          cursor: pointer;
          transition: all 0.2s;
          box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .destinely-got-it-btn:hover {
          background: #f0fdf4;
          transform: translateY(-1px);
        }
        
        .destinely-dashboard-link {
          margin: 16px 0;
        }
        
        .destinely-dashboard-btn {
          display: inline-block;
          background: #10b981;
          color: white;
          text-decoration: none;
          padding: 12px 24px;
          border-radius: 8px;
          font-weight: 600;
          font-size: 14px;
          transition: all 0.2s;
          box-shadow: 0 2px 4px rgba(16, 185, 129, 0.3);
        }
        
        .destinely-dashboard-btn:hover {
          background: #059669;
          transform: translateY(-1px);
          box-shadow: 0 4px 8px rgba(16, 185, 129, 0.4);
        }
        
        @keyframes destinely-fade-in {
          from { opacity: 0; }
          to { opacity: 1; }
        }
        
        @keyframes destinely-scale-in {
          from { 
            opacity: 0;
            transform: translate(-50%, -50%) scale(0.9);
          }
          to { 
            opacity: 1;
            transform: translate(-50%, -50%) scale(1);
          }
        }
        
        @keyframes destinely-ping {
          0% {
            transform: scale(1);
            opacity: 1;
          }
          75%, 100% {
            transform: scale(2);
            opacity: 0;
          }
        }
        
        @keyframes destinely-bounce {
          0%, 20%, 53%, 80%, 100% {
            transform: translate3d(0,0,0);
          }
          40%, 43% {
            transform: translate3d(0, -8px, 0);
          }
          70% {
            transform: translate3d(0, -4px, 0);
          }
          90% {
            transform: translate3d(0, -2px, 0);
          }
        }
      </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', styles);
  }

  function trackEvent(eventType, data) {
    if (!window.destinelyConfig) return;
    
    const payload = {
      hotel_id: window.destinelyConfig.hotelId,
      event_type: eventType,
      timestamp: new Date().toISOString(),
      page_url: window.location.href,
      user_agent: navigator.userAgent,
      ...data
    };
    
    // Send tracking data to analytics endpoint
    fetch(`${window.destinelyConfig.apiBase}/api/widget/track`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload)
    }).catch(error => {
      console.error('Failed to track event:', error);
    });
  }
  
  // Expose widget functions globally for WordPress integration
  window.DestinelyWidget = {
    openModal: window.openDestinelyModal,
    closeModal: window.closeDestinelyModal,
    trackEvent: trackEvent
  };
  
})();
