<?php
/**
 * Analytics class
 * 
 * Handles analytics tracking and reporting
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DestinelyAnalytics {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_analytics_menu'));
        add_action('wp_ajax_destinely_get_analytics', array($this, 'get_analytics_ajax'));
    }
    
    /**
     * Add analytics menu
     */
    public function add_analytics_menu() {
        add_submenu_page(
            'options-general.php',
            __('Destinely Analytics', 'destinely-experiences-widget'),
            __('Destinely Analytics', 'destinely-experiences-widget'),
            'manage_options',
            'destinely-analytics',
            array($this, 'analytics_page')
        );
    }
    
    /**
     * Create analytics table
     */
    public function create_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            hotel_id varchar(255) NOT NULL,
            event_type varchar(50) NOT NULL,
            session_id varchar(255) NOT NULL,
            page_url text NOT NULL,
            page_title varchar(255) NOT NULL,
            user_agent text NOT NULL,
            ip_address varchar(45) NOT NULL,
            event_data longtext NOT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY hotel_id (hotel_id),
            KEY event_type (event_type),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Analytics page
     */
    public function analytics_page() {
        $settings = DestinelyWidget::get_settings();
        $hotel_id = $settings['hotel_id'] ?? '';
        
        if (empty($hotel_id)) {
            echo '<div class="wrap"><h1>' . esc_html__('Destinely Analytics', 'destinely-experiences-widget') . '</h1>';
            echo '<div class="notice notice-warning"><p>' . esc_html__('Please configure your Hotel ID in the Destinely Widget settings to view analytics.', 'destinely-experiences-widget') . '</p></div></div>';
            return;
        }
        
        $analytics_data = $this->get_analytics_data($hotel_id);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Destinely Analytics', 'destinely-experiences-widget'); ?></h1>
            
            <div class="destinely-analytics-header">
                <div class="destinely-analytics-info">
                    <h2><?php esc_html_e('Widget Performance', 'destinely-experiences-widget'); ?></h2>
                    <p><?php esc_html_e('Track how your Destinely widget is performing on your website.', 'destinely-experiences-widget'); ?></p>
                </div>
                <div class="destinely-analytics-actions">
                    <button type="button" class="button button-secondary" id="refresh-analytics">
                        <?php esc_html_e('Refresh Data', 'destinely-experiences-widget'); ?>
                    </button>
                    <button type="button" class="button button-primary" id="export-analytics">
                        <?php esc_html_e('Export CSV', 'destinely-experiences-widget'); ?>
                    </button>
                </div>
            </div>
            
            <div class="destinely-analytics-content">
                <div class="destinely-analytics-overview">
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Widget Loads', 'destinely-experiences-widget'); ?></h3>
                        <div class="analytics-number"><?php echo number_format($analytics_data['widget_loads']); ?></div>
                        <div class="analytics-change <?php echo $analytics_data['loads_change'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $analytics_data['loads_change'] >= 0 ? '+' : ''; ?><?php echo number_format($analytics_data['loads_change'], 1); ?>%
                        </div>
                    </div>
                    
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Widget Interactions', 'destinely-experiences-widget'); ?></h3>
                        <div class="analytics-number"><?php echo number_format($analytics_data['widget_interactions']); ?></div>
                        <div class="analytics-change <?php echo $analytics_data['interactions_change'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $analytics_data['interactions_change'] >= 0 ? '+' : ''; ?><?php echo number_format($analytics_data['interactions_change'], 1); ?>%
                        </div>
                    </div>
                    
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Bookings', 'destinely-experiences-widget'); ?></h3>
                        <div class="analytics-number"><?php echo number_format($analytics_data['bookings']); ?></div>
                        <div class="analytics-change <?php echo $analytics_data['bookings_change'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $analytics_data['bookings_change'] >= 0 ? '+' : ''; ?><?php echo number_format($analytics_data['bookings_change'], 1); ?>%
                        </div>
                    </div>
                    
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Revenue', 'destinely-experiences-widget'); ?></h3>
                        <div class="analytics-number">$<?php echo number_format($analytics_data['revenue'], 2); ?></div>
                        <div class="analytics-change <?php echo $analytics_data['revenue_change'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $analytics_data['revenue_change'] >= 0 ? '+' : ''; ?><?php echo number_format($analytics_data['revenue_change'], 1); ?>%
                        </div>
                    </div>
                </div>
                
                <div class="destinely-analytics-charts">
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Widget Performance Over Time', 'destinely-experiences-widget'); ?></h3>
                        <div id="performance-chart"></div>
                    </div>
                    
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Top Performing Pages', 'destinely-experiences-widget'); ?></h3>
                        <div class="top-pages-list">
                            <?php foreach ($analytics_data['top_pages'] as $page) : ?>
                                <div class="page-item">
                                    <div class="page-title"><?php echo esc_html($page['title']); ?></div>
                                    <div class="page-stats">
                                        <span class="page-loads"><?php echo number_format($page['loads']); ?> loads</span>
                                        <span class="page-interactions"><?php echo number_format($page['interactions']); ?> interactions</span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                
                <div class="destinely-analytics-details">
                    <div class="destinely-analytics-card">
                        <h3><?php esc_html_e('Recent Events', 'destinely-experiences-widget'); ?></h3>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e('Event', 'destinely-experiences-widget'); ?></th>
                                    <th><?php esc_html_e('Page', 'destinely-experiences-widget'); ?></th>
                                    <th><?php esc_html_e('Date', 'destinely-experiences-widget'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($analytics_data['recent_events'] as $event) : ?>
                                    <tr>
                                        <td><?php echo esc_html($event['event_type']); ?></td>
                                        <td><?php echo esc_html($event['page_title']); ?></td>
                                        <td><?php echo esc_html($event['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Initialize charts
            if (typeof Chart !== 'undefined') {
                var ctx = document.getElementById('performance-chart').getContext('2d');
                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: <?php echo json_encode($analytics_data['chart_labels']); ?>,
                        datasets: [{
                            label: 'Widget Loads',
                            data: <?php echo json_encode($analytics_data['chart_loads']); ?>,
                            borderColor: '#10B981',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            tension: 0.4
                        }, {
                            label: 'Interactions',
                            data: <?php echo json_encode($analytics_data['chart_interactions']); ?>,
                            borderColor: '#3B82F6',
                            backgroundColor: 'rgba(59, 130, 246, 0.1)',
                            tension: 0.4
                        }]
                    },
                    options: {
                        responsive: true,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }
            
            // Refresh analytics
            $('#refresh-analytics').on('click', function() {
                location.reload();
            });
            
            // Export analytics
            $('#export-analytics').on('click', function() {
                window.location.href = '<?php echo esc_url(admin_url('admin-ajax.php')); ?>?action=destinely_export_analytics&hotel_id=<?php echo esc_attr($hotel_id); ?>&nonce=<?php echo esc_attr(wp_create_nonce('destinely_export_nonce')); ?>';
            });
        });
        </script>
        <?php
    }
    
    /**
     * Get analytics data
     */
    private function get_analytics_data($hotel_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        // Get current period data (last 30 days)
        $current_start = date('Y-m-d H:i:s', strtotime('-30 days'));
        $current_data = $this->get_period_data($hotel_id, $current_start);
        
        // Get previous period data (30-60 days ago)
        $previous_start = date('Y-m-d H:i:s', strtotime('-60 days'));
        $previous_end = date('Y-m-d H:i:s', strtotime('-30 days'));
        $previous_data = $this->get_period_data($hotel_id, $previous_start, $previous_end);
        
        // Calculate changes
        $loads_change = $previous_data['widget_loads'] > 0 ? (($current_data['widget_loads'] - $previous_data['widget_loads']) / $previous_data['widget_loads']) * 100 : 0;
        $interactions_change = $previous_data['widget_interactions'] > 0 ? (($current_data['widget_interactions'] - $previous_data['widget_interactions']) / $previous_data['widget_interactions']) * 100 : 0;
        $bookings_change = $previous_data['bookings'] > 0 ? (($current_data['bookings'] - $previous_data['bookings']) / $previous_data['bookings']) * 100 : 0;
        $revenue_change = $previous_data['revenue'] > 0 ? (($current_data['revenue'] - $previous_data['revenue']) / $previous_data['revenue']) * 100 : 0;
        
        // Get chart data
        $chart_data = $this->get_chart_data($hotel_id);
        
        // Get top pages
        $top_pages = $this->get_top_pages($hotel_id);
        
        // Get recent events
        $recent_events = $this->get_recent_events($hotel_id);
        
        return array(
            'widget_loads' => $current_data['widget_loads'],
            'widget_interactions' => $current_data['widget_interactions'],
            'bookings' => $current_data['bookings'],
            'revenue' => $current_data['revenue'],
            'loads_change' => $loads_change,
            'interactions_change' => $interactions_change,
            'bookings_change' => $bookings_change,
            'revenue_change' => $revenue_change,
            'chart_labels' => $chart_data['labels'],
            'chart_loads' => $chart_data['loads'],
            'chart_interactions' => $chart_data['interactions'],
            'top_pages' => $top_pages,
            'recent_events' => $recent_events
        );
    }
    
    /**
     * Get period data
     */
    private function get_period_data($hotel_id, $start_date, $end_date = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $where_clause = $wpdb->prepare("hotel_id = %s AND created_at >= %s", $hotel_id, $start_date);
        if ($end_date) {
            $where_clause .= $wpdb->prepare(" AND created_at < %s", $end_date);
        }
        
        $widget_loads = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where_clause AND event_type = 'widget_load'");
        $widget_interactions = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where_clause AND event_type = 'widget_interaction'");
        $bookings = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where_clause AND event_type = 'booking'");
        
        // For revenue, we'd need to join with booking data
        $revenue = 0; // Placeholder - would need actual booking data
        
        return array(
            'widget_loads' => (int) $widget_loads,
            'widget_interactions' => (int) $widget_interactions,
            'bookings' => (int) $bookings,
            'revenue' => (float) $revenue
        );
    }
    
    /**
     * Get chart data
     */
    private function get_chart_data($hotel_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $labels = array();
        $loads = array();
        $interactions = array();
        
        // Get last 7 days
        for ($i = 6; $i >= 0; $i--) {
            $date = date('Y-m-d', strtotime("-$i days"));
            $labels[] = date('M j', strtotime($date));
            
            $day_start = $date . ' 00:00:00';
            $day_end = $date . ' 23:59:59';
            
            $day_loads = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE hotel_id = %s AND event_type = 'widget_load' AND created_at BETWEEN %s AND %s",
                $hotel_id, $day_start, $day_end
            ));
            
            $day_interactions = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE hotel_id = %s AND event_type = 'widget_interaction' AND created_at BETWEEN %s AND %s",
                $hotel_id, $day_start, $day_end
            ));
            
            $loads[] = (int) $day_loads;
            $interactions[] = (int) $day_interactions;
        }
        
        return array(
            'labels' => $labels,
            'loads' => $loads,
            'interactions' => $interactions
        );
    }
    
    /**
     * Get top pages
     */
    private function get_top_pages($hotel_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT page_title, page_url, 
                    COUNT(CASE WHEN event_type = 'widget_load' THEN 1 END) as loads,
                    COUNT(CASE WHEN event_type = 'widget_interaction' THEN 1 END) as interactions
             FROM $table_name 
             WHERE hotel_id = %s 
             GROUP BY page_url 
             ORDER BY loads DESC 
             LIMIT 10",
            $hotel_id
        ));
        
        return $results;
    }
    
    /**
     * Get recent events
     */
    private function get_recent_events($hotel_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT event_type, page_title, created_at 
             FROM $table_name 
             WHERE hotel_id = %s 
             ORDER BY created_at DESC 
             LIMIT 20",
            $hotel_id
        ));
        
        return $results;
    }
    
    /**
     * AJAX handler for analytics
     */
    public function get_analytics_ajax() {
        check_ajax_referer('destinely_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'destinely-experiences-widget'));
        }
        
        $settings = DestinelyWidget::get_settings();
        $hotel_id = $settings['hotel_id'] ?? '';
        
        if (empty($hotel_id)) {
            wp_send_json_error(esc_html__('Hotel ID not configured', 'destinely-experiences-widget'));
        }
        
        $analytics_data = $this->get_analytics_data($hotel_id);
        wp_send_json_success($analytics_data);
    }
} 