<?php
/**
 * Core widget class
 * 
 * Handles widget loading and integration with the Destinely platform
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DestinelyWidgetCore {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'load_widget'));
    }
    
      /**
   * Enqueue scripts - FULLY INTEGRATED with main platform
   */
  public function enqueue_scripts() {
    // Only enqueue on booking confirmation pages
    if (!$this->is_booking_confirmation_page()) {
      return;
    }
    
    // Enqueue widget script - FULLY INTEGRATED with main platform
    wp_enqueue_script(
      'destinely-experiences-widget',
      DestinelyWidget::get_api_base_url() . '/widget/embed.js',
      array(),
      DESTINELY_WIDGET_VERSION,
      true
    );
    
    // Enqueue local widget script with DPO payment integration
    wp_enqueue_script(
      'destinely-widget-local',
      plugin_dir_url(__FILE__) . '../assets/js/widget.js',
      array('destinely-experiences-widget'),
      DESTINELY_WIDGET_VERSION,
      true
    );
    
    // Enqueue real-time updates script
    wp_enqueue_script(
      'destinely-realtime',
      DestinelyWidget::get_api_base_url() . '/widget/realtime.js',
      array('destinely-experiences-widget'),
      DESTINELY_WIDGET_VERSION,
      true
    );
    
    // Localize script with FULL platform integration
    $settings = DestinelyWidget::get_settings();
    wp_localize_script('destinely-experiences-widget', 'destinelyConfig', array(
      'hotelId' => $settings['hotel_id'],
      'color' => $settings['widget_color'],
      'title' => $settings['widget_title'],
      'apiBase' => DestinelyWidget::get_api_base_url(),
      'assetBaseUrl' => DestinelyWidget::get_api_base_url(),
      'supabaseUrl' => DestinelyWidget::get_supabase_url(),
      'supabaseKey' => DestinelyWidget::get_supabase_anon_key(),
      'realtimeEnabled' => true,
      'notificationsEnabled' => true,
      'branding' => array(
        'logo' => DestinelyWidget::get_api_base_url() . '/images/destinely-logo.png',
        'colors' => array(
          'primary' => $settings['widget_color'] ?: '#2A9D8F',
          'secondary' => '#3B82F6',
          'accent' => '#F59E0B'
        ),
        'fonts' => array(
          'primary' => '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif'
        )
      ),
      'features' => array(
        'realTimeUpdates' => true,
        'notifications' => true,
        'analytics' => true,
        'bookingIntegration' => true,
        'paymentProcessing' => true,
        'dpoPaymentIntegration' => true,
        'securePaymentRedirect' => true
      )
    ));
  }
    
      /**
   * Load widget - EXACTLY like the main widget
   */
  public function load_widget() {
    // Only load on booking confirmation pages
    if (!$this->is_booking_confirmation_page()) {
      return;
    }
    
    $settings = DestinelyWidget::get_settings();
    
    // Track widget load - EXACTLY like the widget
    $this->track_event('widget_load', array(
      'hotel_id' => $settings['hotel_id'],
      'page_url' => get_permalink(),
      'page_title' => get_the_title()
    ));
    
    // Fire action for other plugins/themes
    do_action('destinely_widget_loaded', $settings['hotel_id']);
    
    // Ensure widget data is fetched immediately
    $this->ensure_widget_data_fetched();
  }
  
  /**
   * Ensure widget data is fetched immediately - FULLY INTEGRATED with main platform
   */
  private function ensure_widget_data_fetched() {
    $settings = DestinelyWidget::get_settings();
    
    if (empty($settings['hotel_id'])) {
      return;
    }
    
    // Fetch widget data immediately to ensure it's available - FULLY INTEGRATED
    $widget_data = $this->get_widget_settings();
    
    if ($widget_data && isset($widget_data['experiences']) && !empty($widget_data['experiences'])) {
      // Log successful data fetch
      if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('Destinely Widget: Successfully fetched ' . count($widget_data['experiences']) . ' experiences for hotel ' . $settings['hotel_id']);
      }
      
      // Track successful data fetch - FULLY INTEGRATED with main platform
      $this->track_event('widget_data_fetched', null, array(
        'experience_count' => count($widget_data['experiences']),
        'hotel_id' => $settings['hotel_id'],
        'source' => 'wordpress-plugin',
        'platform_integration' => 'full'
      ));
      
      // Initialize real-time updates for experiences
      $this->initialize_realtime_updates($widget_data['experiences']);
      
      // Initialize notification system
      $this->initialize_notifications();
      
    } else {
      // Log failed data fetch
      if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('Destinely Widget: Failed to fetch experiences for hotel ' . $settings['hotel_id']);
      }
      
      // Track failed data fetch
      $this->track_event('widget_data_fetch_failed', null, array(
        'hotel_id' => $settings['hotel_id'],
        'source' => 'wordpress-plugin'
      ));
    }
  }
  
  /**
   * Initialize real-time updates - FULLY INTEGRATED with main platform
   */
  private function initialize_realtime_updates($experiences) {
    // This will be handled by the realtime.js script
    // The plugin ensures the Supabase connection is available
    if (defined('WP_DEBUG') && WP_DEBUG) {
      error_log('Destinely Widget: Initializing real-time updates for ' . count($experiences) . ' experiences');
    }
  }
  
  /**
   * Initialize notification system - FULLY INTEGRATED with main platform
   */
  private function initialize_notifications() {
    // This will be handled by the notification system in the main platform
    // The plugin ensures notifications are enabled
    if (defined('WP_DEBUG') && WP_DEBUG) {
      error_log('Destinely Widget: Initializing notification system');
    }
  }
    
    /**
     * Check if current page is booking confirmation
     */
    private function is_booking_confirmation_page() {
        global $post;
        
        // Allow filtering
        $is_booking = apply_filters('destinely_is_booking_page', false, $post);
        if ($is_booking) {
            return true;
        }
        
        // Check URL patterns
        $current_url = $_SERVER['REQUEST_URI'];
        $booking_patterns = array(
            '/booking-confirmation',
            '/thank-you',
            '/confirmation',
            '/success',
            '/booking-success',
            '/reservation-confirmation'
        );
        
        foreach ($booking_patterns as $pattern) {
            if (strpos($current_url, $pattern) !== false) {
                return true;
            }
        }
        
        // Check page title
        if ($post && $post->post_title) {
            $title = strtolower($post->post_title);
            $booking_keywords = array(
                'booking confirmation',
                'thank you',
                'confirmation',
                'success',
                'booking success',
                'reservation confirmation'
            );
            
            foreach ($booking_keywords as $keyword) {
                if (strpos($title, $keyword) !== false) {
                    return true;
                }
            }
        }
        
        // Check page template
        if ($post && get_page_template_slug($post->ID)) {
            $template = get_page_template_slug($post->ID);
            if (strpos($template, 'booking') !== false || strpos($template, 'confirmation') !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Track widget event
     */
    private function track_event($event_type, $data = array()) {
        $settings = DestinelyWidget::get_settings();
        
        $event_data = array_merge($data, array(
            'hotel_id' => $settings['hotel_id'],
            'event_type' => $event_type,
            'session_id' => $this->get_session_id(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip_address' => $this->get_client_ip(),
            'page_url' => get_permalink(),
            'page_title' => get_the_title(),
            'timestamp' => current_time('mysql')
        ));
        
        // Send to Destinely API
        $this->send_tracking_data($event_data);
        
        // Store locally for analytics
        $this->store_local_analytics($event_data);
    }
    
    /**
     * Get session ID
     */
    private function get_session_id() {
        if (!session_id()) {
            session_start();
        }
        
        if (!isset($_SESSION['destinely_session_id'])) {
            $_SESSION['destinely_session_id'] = 'sess_' . uniqid() . '_' . time();
        }
        
        return $_SESSION['destinely_session_id'];
    }
    
    /**
     * Get client IP
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '';
    }
    
      /**
   * Send tracking data to Destinely API - EXACTLY like the main widget
   */
  private function send_tracking_data($data) {
    $api_url = DestinelyWidget::get_api_base_url() . '/api/widget/track';
    
    // Use EXACTLY the same tracking format as the widget
    $tracking_data = array(
      'hotel_id' => $data['hotel_id'],
      'experience_id' => $data['experience_id'] ?? null,
      'event_type' => $data['event_type'],
      'session_id' => $data['session_id'],
      'metadata' => array_merge($data, array(
        'timestamp' => date('c'),
        'source' => 'wordpress-plugin'
      ))
    );
    
    $response = wp_remote_post($api_url, array(
      'headers' => array(
        'Content-Type' => 'application/json',
        'X-Session-ID' => $data['session_id'],
        'User-Agent' => 'Destinely-WordPress-Plugin/' . DESTINELY_WIDGET_VERSION
      ),
      'body' => json_encode($tracking_data),
      'timeout' => 5,
      'blocking' => false // Don't block page load
    ));
    
    if (is_wp_error($response)) {
      DestinelyWidget::log_error('Failed to send tracking data', array(
        'error' => $response->get_error_message(),
        'data' => $tracking_data
      ));
    }
  }
    
    /**
     * Store analytics locally
     */
    private function store_local_analytics($data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'destinely_analytics';
        
        $wpdb->insert(
            $table_name,
            array(
                'hotel_id' => $data['hotel_id'],
                'event_type' => $data['event_type'],
                'session_id' => $data['session_id'],
                'page_url' => $data['page_url'],
                'page_title' => $data['page_title'],
                'user_agent' => $data['user_agent'],
                'ip_address' => $data['ip_address'],
                'event_data' => json_encode($data),
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
        );
    }
    
      /**
   * Get widget settings from API - EXACTLY like the main widget
   */
  public function get_widget_settings() {
    $settings = DestinelyWidget::get_settings();
    
    if (empty($settings['hotel_id'])) {
      return false;
    }
    
    // Generate session ID exactly like the widget
    $session_id = 'sess_' . substr(md5(uniqid()), 0, 9) . '_' . time();
    
    // Use EXACTLY the same API call as the widget
    $api_url = DestinelyWidget::get_api_base_url() . '/api/widget/config/' . $settings['hotel_id'];
    
    $response = wp_remote_get($api_url, array(
      'timeout' => 10,
      'headers' => array(
        'X-Session-ID' => $session_id,
        'User-Agent' => 'Destinely-WordPress-Plugin/' . DESTINELY_WIDGET_VERSION
      )
    ));
    
    if (is_wp_error($response)) {
      DestinelyWidget::log_error('Failed to get widget settings', array(
        'error' => $response->get_error_message(),
        'hotel_id' => $settings['hotel_id']
      ));
      return false;
    }
    
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    
    // Check for success and experiences exactly like the widget
    if (!$data || !isset($data['success']) || !$data['success'] || !isset($data['experiences']) || empty($data['experiences'])) {
      DestinelyWidget::log_error('Invalid widget settings response', array(
        'response' => $body,
        'hotel_id' => $settings['hotel_id']
      ));
      return false;
    }
    
    // Process experiences to ensure proper image handling
    if (isset($data['experiences']) && is_array($data['experiences'])) {
      foreach ($data['experiences'] as &$experience) {
        // Ensure images are properly formatted for the carousel
        if (isset($experience['images']) && is_array($experience['images'])) {
          // Keep the images array as is - it supports multiple images
          $experience['images'] = array_filter($experience['images'], function($img) {
            return !empty($img) && filter_var($img, FILTER_VALIDATE_URL);
          });
        } elseif (isset($experience['image_url']) && !empty($experience['image_url'])) {
          // Convert single image_url to images array for consistency
          $experience['images'] = array($experience['image_url']);
          unset($experience['image_url']);
        } else {
          // Set default placeholder
          $experience['images'] = array(DestinelyWidget::get_api_base_url() . '/images/experience-placeholder.jpg');
        }
        
        // Ensure we have at least one image
        if (empty($experience['images'])) {
          $experience['images'] = array(DestinelyWidget::get_api_base_url() . '/images/experience-placeholder.jpg');
        }
        
        // Limit to 3 images for carousel (as requested)
        $experience['images'] = array_slice($experience['images'], 0, 3);
      }
    }
    
    return $data;
  }
    
    /**
     * Test widget connection
     */
    public function test_connection() {
        $settings = DestinelyWidget::get_settings();
        
        if (empty($settings['hotel_id'])) {
            return array(
                'success' => false,
                'message' => 'Hotel ID not configured'
            );
        }
        
        $widget_data = $this->get_widget_settings();
        
        if (!$widget_data) {
            return array(
                'success' => false,
                'message' => 'Failed to connect to Destinely API'
            );
        }
        
        return array(
            'success' => true,
            'message' => 'Connection successful',
            'data' => $widget_data
        );
    }
} 