/**
 * Destinely Widget Admin JavaScript - Enhanced Version 2.1 with Google Maps Integration
 */

(function($) {
    'use strict';

    // Initialize admin functionality
    $(document).ready(function() {
        initAdminFeatures();
        setupFormValidation();
        setupColorPicker();
        setupPreviewWidget();
        setupHelpTooltips();
    });

    /**
     * Initialize admin features
     */
    function initAdminFeatures() {
        // Add loading states to form submission
        $('#destinely-widget-settings-form').on('submit', function() {
            $(this).addClass('destinely-loading');
        });

        // Auto-save functionality
        $('input, select').on('change', function() {
            if ($(this).attr('name') !== 'destinely_hotel_id' && $(this).attr('name') !== 'destinely_hotel_name') {
                autoSave();
            }
        });

        // Real-time validation
        $('#destinely_hotel_id').on('blur', validateHotelId);
        $('#destinely_hotel_name').on('blur', validateHotelName);
    }

    /**
     * Setup form validation
     */
    function setupFormValidation() {
        const form = $('#destinely-widget-settings-form');
        
        form.on('submit', function(e) {
            let isValid = true;
            let errorMessage = '';

            // Validate required fields
            const hotelId = $('#destinely_hotel_id').val().trim();
            const hotelName = $('#destinely_hotel_name').val().trim();

            if (!hotelId) {
                showFieldError('destinely_hotel_id', 'Hotel ID is required');
                isValid = false;
            } else if (!isValidHotelId(hotelId)) {
                showFieldError('destinely_hotel_id', 'Hotel ID must be a valid UUID format');
                isValid = false;
            } else {
                clearFieldError('destinely_hotel_id');
            }

            if (!hotelName) {
                showFieldError('destinely_hotel_name', 'Hotel Name is required');
                isValid = false;
            } else if (hotelName.length < 2) {
                showFieldError('destinely_hotel_name', 'Hotel Name must be at least 2 characters');
                isValid = false;
            } else {
                clearFieldError('destinely_hotel_name');
            }

            if (!isValid) {
                e.preventDefault();
                showNotification('Please fix the errors above before saving.', 'error');
                return false;
            }

            // Show success message
            showNotification('Settings saved successfully!', 'success');
        });
    }

    /**
     * Setup color picker enhancement
     */
    function setupColorPicker() {
        const colorInput = $('input[name="destinely_widget_color"]');
        
        // Add color preview
        colorInput.after('<div class="destinely-color-preview" style="display: inline-block; width: 30px; height: 30px; border: 1px solid #ddd; border-radius: 4px; margin-left: 10px; vertical-align: middle;"></div>');
        
        const preview = $('.destinely-color-preview');
        
        // Update preview on color change
        colorInput.on('change', function() {
            preview.css('background-color', $(this).val());
        });
        
        // Initialize preview
        preview.css('background-color', colorInput.val());
    }

    /**
     * Setup widget preview
     */
    function setupPreviewWidget() {
        // Add preview section
        $('.destinely-help-card').before(`
            <div class="destinely-widget-preview">
                <h4>Widget Preview</h4>
                <div class="destinely-preview-widget" id="destinely-preview">
                    🏨 Discover Local Experiences
                </div>
                <p style="margin-top: 10px; color: #6b7280; font-size: 12px;">
                    This is how your widget will appear to guests
                </p>
            </div>
        `);

        // Update preview on settings change
        $('input, select').on('change', updatePreview);
        updatePreview();
    }

    /**
     * Update widget preview
     */
    function updatePreview() {
        const color = $('input[name="destinely_widget_color"]').val() || '#2A9D8F';
        const title = $('input[name="destinely_widget_title"]').val() || 'Discover Local Experiences';
        const position = $('select[name="destinely_widget_position"]').val() || 'bottom-right';
        
        const preview = $('#destinely-preview');
        preview.css('background-color', color);
        preview.text('🏨 ' + title);
        
        // Update position indicator
        const positionText = position.replace('-', ' ').replace(/\b\w/g, l => l.toUpperCase());
        $('.destinely-widget-preview p').html(`
            This is how your widget will appear to guests<br>
            <small>Position: ${positionText}</small>
        `);
    }

    /**
     * Setup help tooltips
     */
    function setupHelpTooltips() {
        // Add help icons to form fields
        $('.destinely-form-table th').each(function() {
            const fieldName = $(this).text().trim();
            const helpText = getHelpText(fieldName);
            
            if (helpText) {
                $(this).append(`<span class="destinely-help-icon" title="${helpText}" style="margin-left: 5px; color: #6b7280; cursor: help;">❓</span>`);
            }
        });
    }

    /**
     * Get help text for field
     */
    function getHelpText(fieldName) {
        const helpTexts = {
            'Hotel ID': 'Get this from your Destinely dashboard. It\'s a unique identifier for your hotel.',
            'Hotel Name': 'The name that will be displayed to guests in the widget.',
            'Widget Title': 'Customize the header text of the widget.',
            'Widget Color': 'Choose a color that matches your hotel\'s branding.',
            'Widget Position': 'Select where the widget appears on your website.',
            'Auto-Open Widget': 'Automatically show the widget after booking completion.',
            'Trigger Delay': 'How long to wait before showing the widget (in seconds).',
            'Notification Sound': 'Play a sound when the widget appears.',
            'Show Notification Message': 'Display a notification message when the widget appears.'
        };
        
        return helpTexts[fieldName] || '';
    }

    /**
     * Validate hotel ID format
     */
    function isValidHotelId(hotelId) {
        // UUID format validation
        const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
        return uuidRegex.test(hotelId);
    }

    /**
     * Validate hotel ID
     */
    function validateHotelId() {
        const hotelId = $(this).val().trim();
        
        if (hotelId && !isValidHotelId(hotelId)) {
            showFieldError('destinely_hotel_id', 'Hotel ID must be a valid UUID format');
        } else {
            clearFieldError('destinely_hotel_id');
        }
    }

    /**
     * Validate hotel name
     */
    function validateHotelName() {
        const hotelName = $(this).val().trim();
        
        if (hotelName && hotelName.length < 2) {
            showFieldError('destinely_hotel_name', 'Hotel Name must be at least 2 characters');
        } else {
            clearFieldError('destinely_hotel_name');
        }
    }

    /**
     * Show field error
     */
    function showFieldError(fieldName, message) {
        const field = $(`[name="${fieldName}"]`);
        const td = field.closest('td');
        
        // Remove existing error
        td.find('.destinely-field-error').remove();
        
        // Add error message
        td.append(`<div class="destinely-field-error" style="color: #dc2626; font-size: 12px; margin-top: 5px;">${message}</div>`);
        
        // Add error styling
        field.css('border-color', '#dc2626');
    }

    /**
     * Clear field error
     */
    function clearFieldError(fieldName) {
        const field = $(`[name="${fieldName}"]`);
        const td = field.closest('td');
        
        // Remove error message
        td.find('.destinely-field-error').remove();
        
        // Remove error styling
        field.css('border-color', '');
    }

    /**
     * Auto-save functionality
     */
    function autoSave() {
        // Only auto-save non-critical fields
        const autoSaveFields = [
            'destinely_widget_title',
            'destinely_widget_color',
            'destinely_widget_position',
            'destinely_widget_trigger_delay',
            'destinely_widget_auto_open',
            'destinely_widget_sound',
            'destinely_widget_notification'
        ];
        
        const formData = {};
        autoSaveFields.forEach(fieldName => {
            const field = $(`[name="${fieldName}"]`);
            if (field.length) {
                if (field.attr('type') === 'checkbox') {
                    formData[fieldName] = field.is(':checked') ? 1 : 0;
                } else {
                    formData[fieldName] = field.val();
                }
            }
        });
        
        // Send auto-save request
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'destinely_auto_save',
                ...formData,
                nonce: destinely_admin.nonce || ''
            },
            success: function(response) {
                if (response.success) {
                    showNotification('Settings auto-saved', 'success', 2000);
                }
            },
            error: function() {
                console.log('Auto-save failed');
            }
        });
    }

    /**
     * Show notification
     */
    function showNotification(message, type = 'info', duration = 5000) {
        const notification = $(`
            <div class="notice notice-${type} is-dismissible destinely-notification" style="position: fixed; top: 32px; right: 20px; z-index: 999999; max-width: 400px;">
                <p>${message}</p>
                <button type="button" class="notice-dismiss">
                    <span class="screen-reader-text">Dismiss this notice.</span>
                </button>
            </div>
        `);
        
        $('body').append(notification);
        
        // Auto-dismiss
        setTimeout(() => {
            notification.fadeOut(() => notification.remove());
        }, duration);
        
        // Manual dismiss
        notification.find('.notice-dismiss').on('click', function() {
            notification.fadeOut(() => notification.remove());
        });
    }

    /**
     * Test widget functionality
     */
    function testWidget() {
        if (typeof destinely_admin !== 'undefined' && destinely_admin.site_url) {
            const testUrl = destinely_admin.site_url + '?destinely-test=1';
            window.open(testUrl, '_blank');
        }
    }

    // Add test button
    $(document).ready(function() {
        $('.destinely-help-card').append(`
            <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #e2e8f0;">
                <button type="button" class="button button-secondary" onclick="testWidget()" style="margin-right: 10px;">
                    🧪 Test Widget
                </button>
                <button type="button" class="button button-secondary" onclick="window.open('https://www.destinely.com/widget-demo', '_blank')">
                    👀 View Demo
                </button>
            </div>
        `);
    });

})(jQuery);