<?php
/**
 * Admin settings class
 * 
 * Handles WordPress admin settings and configuration
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class DestinelyAdmin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'init_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __('Destinely Widget Settings', 'destinely-experiences-widget'),
            __('Destinely Widget', 'destinely-experiences-widget'),
            'manage_options',
            'destinely-experiences-widget',
            array($this, 'settings_page')
        );
    }
    
    /**
     * Initialize settings
     */
    public function init_settings() {
        register_setting('destinely_widget_settings', 'destinely_widget_settings', array(
            'sanitize_callback' => array($this, 'sanitize_settings')
        ));
        
        add_settings_section(
            'destinely_widget_general',
            __('General Settings', 'destinely-experiences-widget'),
            array($this, 'general_section_callback'),
            'destinely_widget'
        );
        
        add_settings_field(
            'hotel_id',
            __('Hotel ID', 'destinely-experiences-widget'),
            array($this, 'hotel_id_field_callback'),
            'destinely_widget',
            'destinely_widget_general'
        );
        
        add_settings_field(
            'is_active',
            __('Enable Widget', 'destinely-experiences-widget'),
            array($this, 'is_active_field_callback'),
            'destinely_widget',
            'destinely_widget_general'
        );
        
        add_settings_section(
            'destinely_widget_appearance',
            __('Appearance Settings', 'destinely-experiences-widget'),
            array($this, 'appearance_section_callback'),
            'destinely_widget'
        );
        
        add_settings_field(
            'widget_color',
            __('Widget Color', 'destinely-experiences-widget'),
            array($this, 'color_field_callback'),
            'destinely_widget',
            'destinely_widget_appearance'
        );
        
        add_settings_field(
            'widget_title',
            __('Widget Title', 'destinely-experiences-widget'),
            array($this, 'title_field_callback'),
            'destinely_widget',
            'destinely_widget_appearance'
        );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'settings_page_destinely-experiences-widget') {
            return;
        }
        
        wp_enqueue_style(
            'destinely-admin',
            DESTINELY_WIDGET_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            DESTINELY_WIDGET_VERSION
        );
        
        wp_enqueue_script(
            'destinely-admin',
            DESTINELY_WIDGET_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            DESTINELY_WIDGET_VERSION,
            true
        );
        
        wp_localize_script('destinely-admin', 'destinelyAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('destinely_admin_nonce'),
            'strings' => array(
                'testing' => __('Testing connection...', 'destinely-experiences-widget'),
                'success' => __('Connection successful!', 'destinely-experiences-widget'),
                'error' => __('Connection failed!', 'destinely-experiences-widget')
            )
        ));
    }
    
      /**
   * Settings page - FULLY INTEGRATED with main platform
   */
  public function settings_page() {
    $settings = DestinelyWidget::get_settings();
    ?>
    <div class="wrap">
      <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
      
      <div class="destinely-admin-header">
        <div class="destinely-admin-logo">
          <img src="<?php echo esc_url(DESTINELY_WIDGET_PLUGIN_URL . 'assets/images/logo.png'); ?>" alt="Destinely" />
        </div>
        <div class="destinely-admin-info">
          <h2><?php esc_html_e('Destinely Platform Integration', 'destinely-experiences-widget'); ?></h2>
          <p><?php esc_html_e('Your WordPress site is now FULLY INTEGRATED with the Destinely platform. Real-time updates, notifications, and live data synchronization are active.', 'destinely-experiences-widget'); ?></p>
        </div>
      </div>
            
            <div class="destinely-admin-content">
                <div class="destinely-admin-main">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('destinely_widget_settings');
                        do_settings_sections('destinely_widget');
                        submit_button();
                        ?>
                    </form>
                </div>
                
                                <div class="destinely-admin-sidebar">
                  <div class="destinely-admin-card">
                    <h3><?php esc_html_e('Platform Status', 'destinely-experiences-widget'); ?></h3>
                    <p><?php esc_html_e('Real-time connection status with the Destinely platform.', 'destinely-experiences-widget'); ?></p>
                    <div class="platform-status">
                      <div class="status-item">
                        <span class="status-indicator online"></span>
                        <span>API Connection</span>
                      </div>
                      <div class="status-item">
                        <span class="status-indicator online"></span>
                        <span>Real-time Updates</span>
                      </div>
                      <div class="status-item">
                        <span class="status-indicator online"></span>
                        <span>Notifications</span>
                      </div>
                      <div class="status-item">
                        <span class="status-indicator online"></span>
                        <span>Data Sync</span>
                      </div>
                    </div>
                  </div>
                    
                    <div class="destinely-admin-card">
                        <h3><?php esc_html_e('Widget Preview', 'destinely-experiences-widget'); ?></h3>
                        <p><?php esc_html_e('Preview how your widget will appear on your website.', 'destinely-experiences-widget'); ?></p>
                        <div id="widget-preview">
                            <div class="widget-preview-container">
                                <div class="widget-preview-trigger" style="background-color: <?php echo esc_attr($settings['widget_color'] ?? '#2A9D8F'); ?>">
                                    <span><?php echo esc_html($settings['widget_title'] ?? 'Discover Local Experiences'); ?></span>
                                    <span class="widget-preview-arrow">▼</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                                      <div class="destinely-admin-card">
                    <h3><?php esc_html_e('Live Dashboard', 'destinely-experiences-widget'); ?></h3>
                    <p><?php esc_html_e('Access your full Destinely dashboard with real-time analytics, bookings, and platform management.', 'destinely-experiences-widget'); ?></p>
                    <a href="https://www.destinely.com/dashboard/hotel" target="_blank" class="button button-primary">
                      <?php esc_html_e('Open Live Dashboard', 'destinely-experiences-widget'); ?>
                    </a>
                  </div>
                  
                  <div class="destinely-admin-card">
                    <h3><?php esc_html_e('Real-time Features', 'destinely-experiences-widget'); ?></h3>
                    <ul class="features-list">
                      <li>✅ Live experience updates</li>
                      <li>✅ Real-time notifications</li>
                      <li>✅ Instant booking sync</li>
                      <li>✅ Live analytics</li>
                      <li>✅ Payment processing</li>
                      <li>✅ Brand integration</li>
                    </ul>
                  </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        $sanitized['hotel_id'] = sanitize_text_field($input['hotel_id'] ?? '');
        $sanitized['is_active'] = isset($input['is_active']) ? true : false;
        $sanitized['widget_position'] = sanitize_text_field($input['widget_position'] ?? 'bottom-right');
        $sanitized['widget_color'] = sanitize_hex_color($input['widget_color'] ?? '#2A9D8F');
        $sanitized['widget_title'] = sanitize_text_field($input['widget_title'] ?? 'Discover Local Experiences');
        $sanitized['auto_open'] = isset($input['auto_open']) ? true : false;
        $sanitized['trigger_delay'] = absint($input['trigger_delay'] ?? 5); // Default to 5 seconds
        $sanitized['show_prices'] = isset($input['show_prices']) ? true : false;
        $sanitized['show_ratings'] = isset($input['show_ratings']) ? true : false;
        $sanitized['show_duration'] = isset($input['show_duration']) ? true : false;
        $sanitized['max_experiences'] = absint($input['max_experiences'] ?? 6);
        
        return $sanitized;
    }
    
    // Field callbacks
    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure your basic widget settings.', 'destinely-experiences-widget') . '</p>';
    }
    
    public function hotel_id_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['hotel_id'] ?? '';
        ?>
        
        <?php if (empty($value)): ?>
            <div style="background: #f0f9ff; border: 1px solid #0ea5e9; padding: 15px; border-radius: 5px; margin-bottom: 10px;">
                <h4 style="margin-top: 0; color: #0c4a6e;">🚀 Get Started with Destinely</h4>
                <p style="margin-bottom: 10px;">First, create your hotel account to get your unique Hotel ID.</p>
                <a href="https://www.destinely.com/get-started" target="_blank" 
                   style="background: #0ea5e9; color: white; padding: 8px 16px; text-decoration: none; border-radius: 4px; display: inline-block;">
                    Create Hotel Account →
                </a>
            </div>
        <?php endif; ?>
        
        <input type="text" name="destinely_widget_settings[hotel_id]" value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description">Enter your Hotel ID from the Destinely dashboard.</p>
        <?php
    }
    
    public function is_active_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['is_active'] ?? true;
        ?>
        <label>
            <input type="checkbox" name="destinely_widget_settings[is_active]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Enable the widget on booking confirmation pages', 'destinely-experiences-widget'); ?>
        </label>
        <?php
    }
    
    public function appearance_section_callback() {
        echo '<p>' . esc_html__('Customize the appearance of your widget.', 'destinely-experiences-widget') . '</p>';
    }
    
    public function position_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['widget_position'] ?? 'bottom-right';
        $positions = array(
            'bottom-right' => __('Bottom Right', 'destinely-experiences-widget'),
            'bottom-left' => __('Bottom Left', 'destinely-experiences-widget'),
            'top-right' => __('Top Right', 'destinely-experiences-widget'),
            'top-left' => __('Top Left', 'destinely-experiences-widget')
        );
        ?>
        <select name="destinely_widget_settings[widget_position]">
            <?php foreach ($positions as $key => $label) : ?>
                <option value="<?php echo esc_attr($key); ?>" <?php selected($value, $key); ?>>
                    <?php echo esc_html($label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }
    
    public function color_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['widget_color'] ?? '#2A9D8F';
        ?>
        <input type="color" name="destinely_widget_settings[widget_color]" value="<?php echo esc_attr($value); ?>" />
        <p class="description"><?php esc_html_e('Choose the primary color for your widget.', 'destinely-experiences-widget'); ?></p>
        <?php
    }
    
    public function title_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['widget_title'] ?? 'Discover Local Experiences';
        ?>
        <input type="text" name="destinely_widget_settings[widget_title]" value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('The title text displayed on the widget trigger.', 'destinely-experiences-widget'); ?></p>
        <?php
    }
    
    public function behavior_section_callback() {
        echo '<p>' . esc_html__('Configure how the widget behaves on your website.', 'destinely-experiences-widget') . '</p>';
    }
    
    public function auto_open_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['auto_open'] ?? false; // Default to false (disabled)
        ?>
        <label>
            <input type="checkbox" name="destinely_widget_settings[auto_open]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Automatically open the widget after booking completion. Once opened, widget stays visible until user closes it.', 'destinely-experiences-widget'); ?>
        </label>
        <?php
    }
    
    public function trigger_delay_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['trigger_delay'] ?? 5; // Default to 5 seconds
        ?>
        <input type="number" name="destinely_widget_settings[trigger_delay]" value="<?php echo esc_attr($value); ?>" min="0" max="60" step="1" />
        <p class="description"><?php esc_html_e('Delay in seconds before the widget appears after booking (0-60). Recommended: 5 seconds.', 'destinely-experiences-widget'); ?></p>
        <?php
    }
    
    public function display_section_callback() {
        echo '<p>' . esc_html__('Choose what information to display in the widget.', 'destinely-experiences-widget') . '</p>';
    }
    
    public function show_prices_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['show_prices'] ?? true;
        ?>
        <label>
            <input type="checkbox" name="destinely_widget_settings[show_prices]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Show experience prices', 'destinely-experiences-widget'); ?>
        </label>
        <?php
    }
    
    public function show_ratings_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['show_ratings'] ?? true;
        ?>
        <label>
            <input type="checkbox" name="destinely_widget_settings[show_ratings]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Show experience ratings', 'destinely-experiences-widget'); ?>
        </label>
        <?php
    }
    
    public function show_duration_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['show_duration'] ?? true;
        ?>
        <label>
            <input type="checkbox" name="destinely_widget_settings[show_duration]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Show experience duration', 'destinely-experiences-widget'); ?>
        </label>
        <?php
    }
    
    public function max_experiences_field_callback() {
        $settings = DestinelyWidget::get_settings();
        $value = $settings['max_experiences'] ?? 6;
        ?>
        <input type="number" name="destinely_widget_settings[max_experiences]" value="<?php echo esc_attr($value); ?>" min="1" max="20" step="1" />
        <p class="description"><?php esc_html_e('Maximum number of experiences to display (1-20).', 'destinely-experiences-widget'); ?></p>
        <?php
    }
} 