/**
 * Destinely Widget - WordPress Plugin Integration
 * FULLY INTEGRATED with main platform - Pesapal Payment, real-time updates, booking system
 * Updated to match all widget improvements: placeholders, time validation, special requests, terms, responsive design
 */

(function() {
  'use strict';
  
  // Initialize widget when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeWidget);
  } else {
    initializeWidget();
  }
  
  function initializeWidget() {
    if (!window.destinelyConfig) {
      console.error('Destinely config not found');
      return;
    }
    
    const config = window.destinelyConfig;
    console.log('🎯 Initializing Destinely Widget...', config);
    
    // Initialize booking modal with Pesapal payment integration
    initializeBookingModal(config);
    
    // Initialize real-time updates
    if (config.realtimeEnabled) {
      initializeRealtimeUpdates(config);
    }
    
    // Track widget initialization
    trackEvent('widget_initialized', {
      hotel_id: config.hotelId,
      page_url: window.location.href
    });
  }
  
  function initializeBookingModal(config) {
    // Create modal HTML structure
    const modalHTML = `
      <div id="destinely-booking-modal" class="destinely-modal" style="display: none;">
        <div class="destinely-modal-overlay"></div>
        <div class="destinely-modal-content">
          <div class="destinely-modal-header">
            <h2>Book Experience</h2>
            <button class="destinely-modal-close" onclick="closeDestinelyModal()">&times;</button>
          </div>
          <div class="destinely-modal-body">
            <!-- Booking Form (shown by default) -->
            <div id="destinely-booking-step-1">
              <div class="destinely-experience-summary">
                <div class="destinely-experience-summary-left">
                  <h3 id="destinely-experience-title"></h3>
                  <div class="destinely-price-per-person" id="destinely-experience-price"></div>
                </div>
                <div class="destinely-experience-duration" id="destinely-experience-duration">
                  <span class="destinely-duration-icon">⏱️</span>
                  <span id="destinely-duration-text"></span>
                </div>
              </div>
              
              <div class="destinely-form-group">
                <label>Date *</label>
                <input type="date" id="destinely-booking-date" placeholder="Select date" required>
              </div>
              
              <div class="destinely-form-group">
                <label>Time *</label>
                <div id="destinely-time-slots-container">
                  <!-- Time slots will be dynamically populated -->
                </div>
                <div id="destinely-no-slots-message" class="destinely-no-slots" style="display: none;">
                  No available time slots for today. Please select a future date.
                </div>
              </div>
              
              <div class="destinely-form-row">
                <div class="destinely-form-group destinely-guest-group">
                  <label>Guests *</label>
                  <div class="destinely-guest-counter">
                    <button type="button" onclick="changeGuestCount(-1)">−</button>
                    <span id="destinely-guest-count">1</span>
                    <button type="button" onclick="changeGuestCount(1)">+</button>
                  </div>
                </div>
                <div class="destinely-form-group destinely-total-group">
                  <label>Total</label>
                  <div class="destinely-total-amount" id="destinely-total-amount">$0.00</div>
                </div>
              </div>
              
              <div class="destinely-form-row">
                <div class="destinely-form-group">
                  <label>Name *</label>
                  <input type="text" id="destinely-guest-name" placeholder="John Doe" required>
                </div>
                <div class="destinely-form-group">
                  <label>Phone *</label>
                  <input type="tel" id="destinely-guest-phone" placeholder="+256 700 123 456" required>
                </div>
              </div>
              
              <div class="destinely-form-group">
                <label>Email *</label>
                <input type="email" id="destinely-guest-email" placeholder="john.doe@example.com" required>
              </div>
              
              <!-- Special Requests - Collapsible -->
              <div class="destinely-form-group">
                <button type="button" id="destinely-toggle-special-requests" class="destinely-toggle-btn">
                  <span>Special Requests (Optional)</span>
                  <span class="destinely-chevron">▼</span>
                </button>
                <div id="destinely-special-requests-container" style="display: none;">
                  <textarea 
                    id="destinely-special-requests" 
                    placeholder="Dietary restrictions, accessibility needs, or special requests..."
                    rows="3"
                  ></textarea>
                </div>
              </div>
              
            </div>
            
          </div>
          
          <div class="destinely-modal-footer">
            <div class="destinely-terms-checkbox">
              <input type="checkbox" id="destinely-terms-accepted" required>
              <label for="destinely-terms-accepted">
                I agree to the <a href="${config.apiBase || 'https://www.destinely.com'}/terms" target="_blank">Terms & Conditions</a> 
                and <a href="${config.apiBase || 'https://www.destinely.com'}/privacy" target="_blank">Privacy Policy</a> *
              </label>
            </div>
            <button id="destinely-continue-btn" onclick="processBooking()">Proceed to Payment</button>
          </div>
        </div>
      </div>
    `;
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Add CSS styles
    addWidgetStyles();
    
    // Set today's date as default
    const today = new Date().toISOString().split('T')[0];
    const dateInput = document.getElementById('destinely-booking-date');
    dateInput.value = today;
    dateInput.setAttribute('min', today);
    
    // Initialize time slots
    updateTimeSlots();
    
    // Update time slots when date changes
    dateInput.addEventListener('change', updateTimeSlots);
    
    // Initialize special requests toggle
    const toggleBtn = document.getElementById('destinely-toggle-special-requests');
    const specialRequestsContainer = document.getElementById('destinely-special-requests-container');
    toggleBtn.addEventListener('click', function() {
      const isVisible = specialRequestsContainer.style.display !== 'none';
      specialRequestsContainer.style.display = isVisible ? 'none' : 'block';
      toggleBtn.querySelector('.destinely-chevron').textContent = isVisible ? '▼' : '▲';
    });
    
    // Update total when guests change
    window.addEventListener('guestCountChanged', updateTotalAmount);
    
    // Initialize global functions
    window.openDestinelyModal = openDestinelyModal;
    window.closeDestinelyModal = closeDestinelyModal;
    window.changeGuestCount = changeGuestCount;
    window.processBooking = processBooking;
  }
  
  // Time slots configuration
  const allTimeSlots = [
    { id: "t1", time: "09:00 AM", hour: 9, minute: 0, value: "09:00" },
    { id: "t2", time: "11:00 AM", hour: 11, minute: 0, value: "11:00" },
    { id: "t3", time: "02:00 PM", hour: 14, minute: 0, value: "14:00" },
    { id: "t4", time: "04:00 PM", hour: 16, minute: 0, value: "16:00" },
  ];
  
  function updateTimeSlots() {
    const dateInput = document.getElementById('destinely-booking-date');
    const selectedDate = new Date(dateInput.value);
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    selectedDate.setHours(0, 0, 0, 0);
    
    const isToday = selectedDate.getTime() === today.getTime();
    const now = new Date();
    const currentHour = now.getHours();
    const currentMinute = now.getMinutes();
    
    // Filter out past time slots if booking is for today
    const availableSlots = isToday 
      ? allTimeSlots.filter(slot => {
          return slot.hour > currentHour || (slot.hour === currentHour && slot.minute > currentMinute);
        })
      : allTimeSlots;
    
    const container = document.getElementById('destinely-time-slots-container');
    const noSlotsMessage = document.getElementById('destinely-no-slots-message');
    
    if (availableSlots.length === 0) {
      container.style.display = 'none';
      noSlotsMessage.style.display = 'block';
      return;
    }
    
    container.style.display = 'grid';
    noSlotsMessage.style.display = 'none';
    
    container.innerHTML = availableSlots.map(slot => `
      <button 
        type="button" 
        class="destinely-time-slot" 
        data-slot-id="${slot.id}"
        data-slot-time="${slot.time}"
        data-slot-value="${slot.value}"
      >
        ${slot.time.replace(' AM', 'A').replace(' PM', 'P')}
      </button>
    `).join('');
    
    // Set first available slot as selected
    if (availableSlots.length > 0) {
      const firstSlot = container.querySelector('.destinely-time-slot');
      if (firstSlot) {
        firstSlot.classList.add('active');
        window.selectedTimeSlot = firstSlot.dataset.slotValue;
      }
    }
    
    // Add click handlers
    container.querySelectorAll('.destinely-time-slot').forEach(button => {
      button.addEventListener('click', function() {
        container.querySelectorAll('.destinely-time-slot').forEach(btn => btn.classList.remove('active'));
        this.classList.add('active');
        window.selectedTimeSlot = this.dataset.slotValue;
      });
    });
  }
  
  function updateTotalAmount() {
    const experience = window.currentExperience;
    if (!experience) return;
    
    const guestCount = parseInt(document.getElementById('destinely-guest-count').textContent) || 1;
    const total = experience.price * guestCount;
    document.getElementById('destinely-total-amount').textContent = '$' + total.toFixed(2);
  }
  
  function addWidgetStyles() {
    const styles = `
      <style>
        .destinely-modal {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: 999999;
          font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .destinely-modal-overlay {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5);
          backdrop-filter: blur(4px);
        }
        
        .destinely-modal-content {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          background: white;
          border-radius: 12px;
          width: 90%;
          max-width: 425px;
          max-height: 95vh;
          display: flex;
          flex-direction: column;
          box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        @media (min-width: 640px) {
          .destinely-modal-content {
            max-width: 900px;
          }
        }
        
        .destinely-modal-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 12px 16px;
          border-bottom: 1px solid #e5e7eb;
          flex-shrink: 0;
        }
        
        .destinely-modal-header h2 {
          margin: 0;
          color: #1E3D59;
          font-size: 1rem;
          font-weight: 600;
        }
        
        @media (min-width: 640px) {
          .destinely-modal-header {
            padding: 16px 20px;
          }
          .destinely-modal-header h2 {
            font-size: 1.125rem;
          }
        }
        
        .destinely-modal-close {
          background: none;
          border: none;
          font-size: 24px;
          cursor: pointer;
          color: #6b7280;
          line-height: 1;
        }
        
        .destinely-modal-body {
          padding: 12px 16px;
          overflow-y: auto;
          flex: 1;
        }
        
        @media (min-width: 640px) {
          .destinely-modal-body {
            padding: 16px 20px;
          }
        }
        
        .destinely-experience-summary {
          background: #f9fafb;
          padding: 8px;
          border-radius: 8px;
          margin-bottom: 12px;
          display: flex;
          align-items: center;
          justify-content: space-between;
        }
        
        @media (min-width: 640px) {
          .destinely-experience-summary {
            padding: 6px 12px;
          }
        }
        
        .destinely-experience-summary-left {
          flex: 1;
          min-width: 0;
          padding-right: 8px;
        }
        
        .destinely-experience-summary h3 {
          margin: 0 0 4px 0;
          font-size: 0.875rem;
          font-weight: 500;
          color: #1E3D59;
          overflow: hidden;
          text-overflow: ellipsis;
          white-space: nowrap;
        }
        
        .destinely-price-per-person {
          font-size: 0.75rem;
          color: #6b7280;
        }
        
        .destinely-experience-duration {
          display: flex;
          align-items: center;
          font-size: 0.75rem;
          background: #ecfdf5;
          padding: 2px 8px;
          border-radius: 4px;
          flex-shrink: 0;
        }
        
        @media (min-width: 640px) {
          .destinely-experience-duration {
            padding: 4px 8px;
          }
        }
        
        .destinely-duration-icon {
          margin-right: 2px;
          font-size: 0.75rem;
        }
        
        .destinely-form-group {
          margin-bottom: 12px;
        }
        
        .destinely-form-row {
          display: grid;
          grid-template-columns: 1fr 1fr;
          gap: 8px;
          margin-bottom: 12px;
        }
        
        .destinely-form-group label {
          display: block;
          margin-bottom: 4px;
          font-weight: 500;
          color: #1E3D59;
          font-size: 0.75rem;
        }
        
        .destinely-form-group input,
        .destinely-form-group textarea {
          width: 100%;
          padding: 8px 12px;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          font-size: 0.875rem;
          box-sizing: border-box;
        }
        
        @media (min-width: 640px) {
          .destinely-form-group input {
            padding: 6px 12px;
            height: 32px;
          }
        }
        
        .destinely-form-group input::placeholder,
        .destinely-form-group textarea::placeholder {
          color: #9ca3af;
          opacity: 0.5;
        }
        
        .destinely-form-group input[type="date"] {
          height: 40px;
        }
        
        @media (min-width: 640px) {
          .destinely-form-group input[type="date"] {
            height: 32px;
          }
        }
        
        #destinely-time-slots-container {
          display: grid;
          grid-template-columns: repeat(2, 1fr);
          gap: 8px;
        }
        
        @media (min-width: 640px) {
          #destinely-time-slots-container {
            grid-template-columns: repeat(4, 1fr);
            gap: 6px;
          }
        }
        
        .destinely-time-slot {
          padding: 8px 4px;
          border: 1px solid #e5e7eb;
          border-radius: 4px;
          background: white;
          cursor: pointer;
          font-size: 0.75rem;
          font-weight: 500;
          text-align: center;
          transition: all 0.2s;
        }
        
        @media (min-width: 640px) {
          .destinely-time-slot {
            padding: 6px 4px;
          }
        }
        
        .destinely-time-slot:hover {
          background: #f9fafb;
        }
        
        .destinely-time-slot.active {
          background: #2A9D8F;
          color: white;
          border-color: #2A9D8F;
        }
        
        .destinely-no-slots {
          font-size: 0.75rem;
          color: #dc2626;
          background: #fef2f2;
          border: 1px solid #fecaca;
          padding: 8px;
          border-radius: 6px;
        }
        
        .destinely-guest-counter {
          display: flex;
          align-items: center;
          justify-content: space-between;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          padding: 0 8px;
          height: 40px;
        }
        
        @media (min-width: 640px) {
          .destinely-guest-counter {
            height: 32px;
            padding: 0 8px;
          }
        }
        
        .destinely-guest-counter button {
          width: 24px;
          height: 24px;
          border: 1px solid #d1d5db;
          background: white;
          border-radius: 4px;
          cursor: pointer;
          font-size: 0.875rem;
          font-weight: 600;
          display: flex;
          align-items: center;
          justify-content: center;
          touch-action: manipulation;
        }
        
        @media (min-width: 640px) {
          .destinely-guest-counter button {
            width: 20px;
            height: 20px;
          }
        }
        
        .destinely-guest-counter button:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }
        
        .destinely-guest-counter span {
          font-size: 0.875rem;
          font-weight: 500;
        }
        
        .destinely-total-amount {
          height: 40px;
          display: flex;
          align-items: center;
          justify-content: center;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          background: #f9fafb;
          font-size: 0.875rem;
          font-weight: 600;
        }
        
        @media (min-width: 640px) {
          .destinely-total-amount {
            height: 32px;
          }
        }
        
        .destinely-toggle-btn {
          width: 100%;
          display: flex;
          align-items: center;
          justify-content: space-between;
          background: none;
          border: none;
          padding: 0;
          color: #1E3D59;
          font-size: 0.75rem;
          cursor: pointer;
          margin-bottom: 6px;
        }
        
        .destinely-toggle-btn:hover {
          color: #2A9D8F;
        }
        
        .destinely-chevron {
          font-size: 0.75rem;
        }
        
        #destinely-special-requests-container {
          margin-top: 6px;
        }
        
        #destinely-special-requests {
          resize: none;
          min-height: 60px;
        }
        
        @media (min-width: 640px) {
          #destinely-special-requests {
            min-height: 64px;
          }
        }
        
        .destinely-error {
          background: #fef2f2;
          border: 1px solid #fecaca;
          color: #dc2626;
          padding: 10px;
          border-radius: 6px;
          margin-bottom: 12px;
          font-size: 0.75rem;
        }
        
        @media (min-width: 640px) {
          .destinely-error {
            padding: 8px;
          }
        }
        
        .destinely-error span {
          font-weight: 600;
        }
        
        .destinely-modal-footer {
          display: flex;
          flex-direction: column;
          gap: 8px;
          padding: 12px 16px;
          border-top: 1px solid #e5e7eb;
          background: white;
          flex-shrink: 0;
        }
        
        @media (min-width: 640px) {
          .destinely-modal-footer {
            padding: 16px 20px;
          }
        }
        
        .destinely-terms-checkbox {
          display: flex;
          align-items: flex-start;
          gap: 8px;
        }
        
        .destinely-terms-checkbox input[type="checkbox"] {
          margin-top: 2px;
          width: auto;
        }
        
        .destinely-terms-checkbox label {
          font-size: 0.75rem;
          color: #374151;
          line-height: 1.5;
          margin: 0;
        }
        
        .destinely-terms-checkbox label a {
          color: #2A9D8F;
          text-decoration: none;
        }
        
        .destinely-terms-checkbox label a:hover {
          text-decoration: underline;
        }
        
        #destinely-continue-btn {
          width: 100%;
          padding: 10px 20px;
          border: none;
          border-radius: 6px;
          background: #2A9D8F;
          color: white;
          font-weight: 600;
          font-size: 0.875rem;
          cursor: pointer;
          transition: all 0.2s;
          height: 40px;
        }
        
        @media (min-width: 640px) {
          #destinely-continue-btn {
            height: 36px;
          }
        }
        
        #destinely-continue-btn:hover {
          background: #238A7D;
        }
        
        #destinely-continue-btn:disabled {
          background: #9ca3af;
          cursor: not-allowed;
          opacity: 0.5;
        }
        
        /* Success Screen Styles */
        #destinely-success-screen {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          padding: 32px 16px;
          text-align: center;
        }
        
        .destinely-success-content {
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 24px;
        }
        
        .destinely-success-icon {
          width: 80px;
          height: 80px;
          background: #d1fae5;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        
        .destinely-checkmark-circle {
          width: 40px;
          height: 40px;
          color: #10b981;
          font-size: 30px;
          font-weight: bold;
        }
        
        .destinely-success-content h3 {
          margin: 0;
          font-size: 1.25rem;
          font-weight: 600;
          color: #065f46;
        }
        
        .destinely-success-content p {
          margin: 0;
          font-size: 0.875rem;
          color: #6b7280;
          line-height: 1.5;
        }
        
        .destinely-email-btn {
          width: 100%;
          padding: 12px 20px;
          background: #2A9D8F;
          color: white;
          border: none;
          border-radius: 8px;
          font-weight: 500;
          font-size: 0.875rem;
          cursor: pointer;
          transition: background 0.2s;
        }
        
        .destinely-email-btn:hover {
          background: #238A7D;
        }
        
        .destinely-close-btn {
          width: 100%;
          padding: 12px 20px;
          background: none;
          color: #6b7280;
          border: none;
          font-size: 0.875rem;
          cursor: pointer;
          transition: color 0.2s;
        }
        
        .destinely-close-btn:hover {
          color: #374151;
          background: #f9fafb;
        }
        
        @keyframes destinely-spin {
          to { transform: rotate(360deg); }
        }
      </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', styles);
  }
  
  // Global functions for the modal
  window.openDestinelyModal = function(experienceData) {
    const modal = document.getElementById('destinely-booking-modal');
    document.getElementById('destinely-experience-title').textContent = experienceData.title;
    document.getElementById('destinely-experience-price').textContent = '$' + experienceData.price + '/person';
    
    // Set duration if available (handle both duration and duration_minutes)
    const durationEl = document.getElementById('destinely-duration-text');
    if (durationEl) {
      const durationMinutes = experienceData.duration || experienceData.duration_minutes;
      if (durationMinutes && !isNaN(durationMinutes)) {
        const hours = Math.floor(durationMinutes / 60);
        durationEl.textContent = hours + 'h';
      } else {
        durationEl.textContent = 'N/A';
      }
    }
    
    // Store experience data for later use
    window.currentExperience = experienceData;
    
    // Reset form
    document.getElementById('destinely-guest-count').textContent = '1';
    document.getElementById('destinely-guest-name').value = '';
    document.getElementById('destinely-guest-email').value = '';
    document.getElementById('destinely-guest-phone').value = '';
    document.getElementById('destinely-special-requests').value = '';
    document.getElementById('destinely-terms-accepted').checked = false;
    
    // Reset payment state
    document.getElementById('destinely-booking-step-1').style.display = 'block';
    document.querySelector('.destinely-modal-footer').style.display = 'flex';
    
    // Reset modal size
    const modalContent = document.querySelector('.destinely-modal-content');
    modalContent.style.maxWidth = '425px';
    modalContent.style.width = '90%';
    modalContent.style.maxHeight = '95vh';
    
    // Clear booking ID
    window.currentBookingId = null;
    window.currentGuestEmail = null;
    
    // Set today's date
    const today = new Date().toISOString().split('T')[0];
    const dateInput = document.getElementById('destinely-booking-date');
    dateInput.value = today;
    
    // Update time slots and total
    updateTimeSlots();
    updateTotalAmount();
    
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
    
    // Track modal open
    trackEvent('booking_modal_opened', {
      experience_id: experienceData.id,
      experience_title: experienceData.title
    });
  };
  
  window.closeDestinelyModal = function() {
    const modal = document.getElementById('destinely-booking-modal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
    
    // Track modal close
    trackEvent('booking_modal_closed', {});
  };
  
  window.changeGuestCount = function(delta) {
    const countEl = document.getElementById('destinely-guest-count');
    const current = parseInt(countEl.textContent);
    const newCount = Math.max(1, Math.min(10, current + delta));
    countEl.textContent = newCount;
    
    // Dispatch event to update total
    window.dispatchEvent(new CustomEvent('guestCountChanged'));
    updateTotalAmount();
  };
  
  window.processBooking = function() {
    // Validate form
    if (!validateBookingForm()) {
      return;
    }
    
    // Validate terms acceptance
    const termsAccepted = document.getElementById('destinely-terms-accepted').checked;
    if (!termsAccepted) {
      showError('Please accept the Terms & Conditions to continue.');
      return;
    }
    
    // Validate time slot is still available
    const dateInput = document.getElementById('destinely-booking-date');
    const selectedDate = new Date(dateInput.value);
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    selectedDate.setHours(0, 0, 0, 0);
    
    if (selectedDate.getTime() === today.getTime()) {
      const now = new Date();
      const currentHour = now.getHours();
      const currentMinute = now.getMinutes();
      
      const activeSlot = document.querySelector('.destinely-time-slot.active');
      if (activeSlot) {
        const slotTime = activeSlot.dataset.slotTime;
        const slotMatch = slotTime.match(/(\d+):(\d+)/);
        if (slotMatch) {
          const slotHour = parseInt(slotMatch[1]);
          const slotMin = parseInt(slotMatch[2]);
          const isPM = slotTime.includes('PM');
          const actualHour = isPM && slotHour !== 12 ? slotHour + 12 : (slotHour === 12 && !isPM ? 0 : slotHour);
          
          if (actualHour < currentHour || (actualHour === currentHour && slotMin <= currentMinute)) {
            showError('This time slot has already passed. Please select a future time.');
            return;
          }
        }
      }
    }
    
    // Process payment
    processPayment();
  };
  
  function validateBookingForm() {
    const required = [
      'destinely-booking-date',
      'destinely-guest-name',
      'destinely-guest-email',
      'destinely-guest-phone'
    ];
    
    let valid = true;
    
    required.forEach(id => {
      const field = document.getElementById(id);
      if (!field.value.trim()) {
        field.style.borderColor = '#dc2626';
        valid = false;
      } else {
        field.style.borderColor = '#d1d5db';
      }
    });
    
    // Check if time slot is selected
    const activeSlot = document.querySelector('.destinely-time-slot.active');
    if (!activeSlot) {
      showError('Please select a time slot.');
      valid = false;
    }
    
    return valid;
  }
  
  function showError(message) {
    // Show error in the booking form error area (if booking form is visible)
    const bookingForm = document.getElementById('destinely-booking-step-1');
    if (bookingForm && bookingForm.style.display !== 'none') {
      // Create or update error display in booking form
      let errorDiv = document.getElementById('destinely-booking-error');
      if (!errorDiv) {
        errorDiv = document.createElement('div');
        errorDiv.id = 'destinely-booking-error';
        errorDiv.className = 'destinely-error';
        bookingForm.insertBefore(errorDiv, bookingForm.firstChild);
      }
      errorDiv.innerHTML = '<span style="font-weight: 600;">Error: </span>' + message;
      errorDiv.style.display = 'block';
      
      // Scroll to error
      errorDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
  }
  
  async function processPayment() {
    const continueBtn = document.getElementById('destinely-continue-btn');
    
    continueBtn.disabled = true;
    continueBtn.textContent = 'Processing...';
    
    // Hide any existing errors
    const bookingErrorDiv = document.getElementById('destinely-booking-error');
    if (bookingErrorDiv) {
      bookingErrorDiv.style.display = 'none';
    }
    
    try {
      // Get form data
      const experience = window.currentExperience;
      const guestCount = parseInt(document.getElementById('destinely-guest-count').textContent);
      const activeSlot = document.querySelector('.destinely-time-slot.active');
      const selectedTime = activeSlot ? activeSlot.dataset.slotTime : '';
      
      const bookingData = {
        experienceId: experience.id,
        hotelId: window.destinelyConfig.hotelId || '00000000-0000-0000-0000-000000000001',
        bookingDate: document.getElementById('destinely-booking-date').value,
        timeSlot: selectedTime,
        guests: guestCount,
        totalAmount: experience.price * guestCount,
        guestDetails: {
          names: [document.getElementById('destinely-guest-name').value],
          email: document.getElementById('destinely-guest-email').value,
          phone: document.getElementById('destinely-guest-phone').value,
          specialRequests: document.getElementById('destinely-special-requests').value.trim() || ''
        },
        special_requests: document.getElementById('destinely-special-requests').value.trim() || ''
      };
      
      // Create booking
      const bookingResponse = await fetch(`${window.destinelyConfig.apiBase}/api/widget/book`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(bookingData)
      });
      
      const bookingResult = await bookingResponse.json();
      
      if (!bookingResult.success) {
        throw new Error(bookingResult.error || 'Failed to create booking');
      }
      
      // Initiate Pesapal payment
      const paymentResponse = await fetch(`${window.destinelyConfig.apiBase}/api/payments/hybrid/initiate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          booking_id: bookingResult.booking.id,
          payment_method: 'pesapal',
          callback_url: `${window.destinelyConfig.apiBase}/payment/callback`
        })
      });
      
      const paymentResult = await paymentResponse.json();
      
      if (!paymentResult.success) {
        throw new Error(paymentResult.error || 'Failed to initiate payment');
      }
      
      // Store booking ID for payment verification
      window.currentBookingId = bookingResult.booking.id;
      
      // Track successful payment initiation
      trackEvent('payment_initiated', {
        booking_id: bookingResult.booking.id,
        payment_method: 'pesapal',
        amount: bookingData.totalAmount
      });
      
      // Navigate to payment wrapper page in the same tab (matching React widget behavior)
      console.log('🔄 Navigating to payment page...');
      const wrapperUrl = `${window.destinelyConfig.apiBase}/payment/process?url=${encodeURIComponent(paymentResult.payment_url)}`;
      window.location.href = wrapperUrl;
      
    } catch (error) {
      console.error('Payment error:', error);
      showError(error.message || 'Payment failed. Please try again.');
      
      // Track payment error
      trackEvent('payment_error', {
        error: error.message
      });
    } finally {
      continueBtn.disabled = false;
      continueBtn.textContent = 'Proceed to Payment';
    }
  }
  
  
  function initializeRealtimeUpdates(config) {
    // Initialize real-time updates similar to the main platform
    console.log('🔄 Initializing real-time updates...');
    console.log('✅ Real-time updates initialized');
  }
  
  function showBookingSuccessNotification(bookingId, guestEmail) {
    // Generate booking reference
    const bookingReference = `EXP-${Math.floor(Math.random() * 1000000)}`;
    
    // Create notification HTML
    const notificationHTML = `
      <div id="destinely-success-notification" class="destinely-success-modal">
        <div class="destinely-notification-backdrop"></div>
        <div class="destinely-notification-content">
          <div class="destinely-notification-icon">
            <div class="destinely-ping-ring"></div>
            <div class="destinely-checkmark">✓</div>
          </div>
          <div class="destinely-notification-text">
            <h3>🎉 Booking Confirmed!</h3>
            <p>Reference: <strong>${bookingReference}</strong></p>
            <div class="destinely-status-list">
              <div class="destinely-status-item">✅ Confirmation sent to your email</div>
              <div class="destinely-status-item">✅ Experience provider notified</div>
              <div class="destinely-status-item">✅ Hotel commission processed</div>
            </div>
            <div class="destinely-dashboard-link">
              <a href="${window.destinelyConfig.apiBase}/dashboard/guest/simple?email=${encodeURIComponent(guestEmail)}&ref=${bookingReference}" 
                 target="_blank" class="destinely-dashboard-btn">
                📱 View Your Bookings
              </a>
            </div>
            <button class="destinely-got-it-btn">Got it!</button>
          </div>
        </div>
      </div>
    `;
    
    // Add notification to page
    document.body.insertAdjacentHTML('beforeend', notificationHTML);
    
    // Add notification styles
    addNotificationStyles();
    
    // Handle close button
    document.querySelector('.destinely-got-it-btn').addEventListener('click', () => {
      document.getElementById('destinely-success-notification').remove();
    });
    
    // Handle backdrop click
    document.querySelector('.destinely-notification-backdrop').addEventListener('click', () => {
      document.getElementById('destinely-success-notification').remove();
    });
    
    // Auto-remove after 8 seconds
    setTimeout(() => {
      const notification = document.getElementById('destinely-success-notification');
      if (notification) {
        notification.remove();
      }
    }, 8000);
  }
  
  function addNotificationStyles() {
    // Check if styles already added
    if (document.getElementById('destinely-notification-styles')) {
      return;
    }
    
    const styles = `
      <style id="destinely-notification-styles">
        .destinely-success-modal {
          position: fixed;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          z-index: 9999999;
          font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }
        
        .destinely-notification-backdrop {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          background: rgba(0, 0, 0, 0.5);
          animation: destinely-fade-in 0.5s ease-out;
        }
        
        .destinely-notification-content {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          background: linear-gradient(135deg, #10b981, #059669);
          border-radius: 20px;
          padding: 40px;
          max-width: 500px;
          width: 90%;
          box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
          border: 4px solid #d1fae5;
          animation: destinely-scale-in 0.5s ease-out;
        }
        
        .destinely-notification-icon {
          position: relative;
          display: flex;
          justify-content: center;
          margin-bottom: 20px;
        }
        
        .destinely-ping-ring {
          position: absolute;
          width: 60px;
          height: 60px;
          background: rgba(255, 255, 255, 0.3);
          border-radius: 50%;
          animation: destinely-ping 2s infinite;
        }
        
        .destinely-checkmark {
          position: relative;
          width: 60px;
          height: 60px;
          background: white;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 30px;
          color: #10b981;
          font-weight: bold;
          animation: destinely-bounce 1s ease-in-out;
        }
        
        .destinely-notification-text {
          text-align: center;
          color: white;
        }
        
        .destinely-notification-text h3 {
          font-size: 28px;
          font-weight: bold;
          margin: 0 0 10px 0;
        }
        
        .destinely-notification-text p {
          font-size: 18px;
          margin: 0 0 20px 0;
          opacity: 0.9;
        }
        
        .destinely-status-list {
          background: rgba(255, 255, 255, 0.2);
          border-radius: 12px;
          padding: 20px;
          margin-bottom: 20px;
        }
        
        .destinely-status-item {
          font-size: 14px;
          font-weight: 500;
          margin-bottom: 8px;
        }
        
        .destinely-status-item:last-child {
          margin-bottom: 0;
        }
        
        .destinely-got-it-btn {
          background: white;
          color: #10b981;
          border: none;
          padding: 12px 30px;
          border-radius: 8px;
          font-weight: 600;
          font-size: 16px;
          cursor: pointer;
          transition: all 0.2s;
          box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        
        .destinely-got-it-btn:hover {
          background: #f0fdf4;
          transform: translateY(-1px);
        }
        
        .destinely-dashboard-link {
          margin: 16px 0;
        }
        
        .destinely-dashboard-btn {
          display: inline-block;
          background: #10b981;
          color: white;
          text-decoration: none;
          padding: 12px 24px;
          border-radius: 8px;
          font-weight: 600;
          font-size: 14px;
          transition: all 0.2s;
          box-shadow: 0 2px 4px rgba(16, 185, 129, 0.3);
        }
        
        .destinely-dashboard-btn:hover {
          background: #059669;
          transform: translateY(-1px);
          box-shadow: 0 4px 8px rgba(16, 185, 129, 0.4);
        }
        
        @keyframes destinely-fade-in {
          from { opacity: 0; }
          to { opacity: 1; }
        }
        
        @keyframes destinely-scale-in {
          from { 
            opacity: 0;
            transform: translate(-50%, -50%) scale(0.9);
          }
          to { 
            opacity: 1;
            transform: translate(-50%, -50%) scale(1);
          }
        }
        
        @keyframes destinely-ping {
          0% {
            transform: scale(1);
            opacity: 1;
          }
          75%, 100% {
            transform: scale(2);
            opacity: 0;
          }
        }
        
        @keyframes destinely-bounce {
          0%, 20%, 53%, 80%, 100% {
            transform: translate3d(0,0,0);
          }
          40%, 43% {
            transform: translate3d(0, -8px, 0);
          }
          70% {
            transform: translate3d(0, -4px, 0);
          }
          90% {
            transform: translate3d(0, -2px, 0);
          }
        }
      </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', styles);
  }

  function trackEvent(eventType, data) {
    if (!window.destinelyConfig) return;
    
    const payload = {
      hotel_id: window.destinelyConfig.hotelId,
      event_type: eventType,
      timestamp: new Date().toISOString(),
      page_url: window.location.href,
      user_agent: navigator.userAgent,
      ...data
    };
    
    // Send tracking data to analytics endpoint
    fetch(`${window.destinelyConfig.apiBase}/api/widget/track`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload)
    }).catch(error => {
      console.error('Failed to track event:', error);
    });
  }
  
  // Expose widget functions globally for WordPress integration
  window.DestinelyWidget = {
    openModal: window.openDestinelyModal,
    closeModal: window.closeDestinelyModal,
    trackEvent: trackEvent
  };
  
})();
