<?php
/**
 * Plugin Name: Destinely Experiences Widget
 * Plugin URI: https://www.destinely.com/wordpress-plugin
 * Description: Add a widget to your hotel website that shows nearby experiences to guests after they complete a booking. Simple setup with just Hotel ID, title, and color customization.
 * Version: 2.7.0
 * Author: Destinely
 * Author URI: https://www.destinely.com/about
 * Text Domain: destinely-experiences-widget
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Load stubs for IDE/static analysis when WordPress is not loaded
if (!function_exists('add_action')) {
    require_once __DIR__ . '/wordpress-stubs.php';
}

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('DESTINELY_WIDGET_VERSION', '2.7.0');
define('DESTINELY_WIDGET_PATH', plugin_dir_path(__FILE__));
define('DESTINELY_WIDGET_URL', plugin_dir_url(__FILE__));

class Destinely_Widget {
    /**
     * Constructor
     */
    public function __construct() {
        // Load plugin text domain for translations
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        
        // Add settings page
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Enqueue scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Add widget to footer
        add_action('wp_footer', array($this, 'add_widget_to_footer'));
        
        // Add activation hook
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // Add deactivation hook
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Add admin notices
        add_action('admin_notices', array($this, 'admin_notices'));
    }

    /**
     * Load plugin text domain for translations
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'destinely-experiences-widget',
            false,
            dirname(plugin_basename(__FILE__)) . '/languages'
        );
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        add_option('destinely_widget_enabled', 1);
        add_option('destinely_widget_auto_open', 0); // Disabled by default - widget stays visible once opened
        add_option('destinely_widget_sound', 1);
        add_option('destinely_widget_notification', 1);
        add_option('destinely_widget_position', 'bottom-right');
        add_option('destinely_widget_trigger_delay', 5); // 5 seconds delay after booking
        add_option('destinely_widget_color', '#2A9D8F');
        add_option('destinely_widget_title', 'Discover Local Experiences');
        add_option('destinely_widget_section_title', 'Nearby Experiences');
        add_option('destinely_widget_size', 'medium');
        add_option('destinely_widget_show_prices', 1);
        add_option('destinely_widget_show_ratings', 1);
        add_option('destinely_widget_show_duration', 1);
        add_option('destinely_widget_max_experiences', 6);
        add_option('destinely_widget_status', 'active');
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up if needed
    }

    /**
     * Admin notices
     */
    public function admin_notices() {
        if (!get_option('destinely_hotel_id')) {
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>' . esc_html__('Destinely Experiences Widget:', 'destinely-experiences-widget') . '</strong> ';
            printf(
                /* translators: %s: Settings page URL */
                esc_html__('Please configure your Hotel ID and Hotel Name in the %s to activate the widget.', 'destinely-experiences-widget'),
                '<a href="' . esc_url(admin_url('options-general.php?page=destinely-widget')) . '">' . esc_html__('settings page', 'destinely-experiences-widget') . '</a>'
            );
            echo '</p></div>';
        }
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        $hook = add_options_page(
            __('Destinely Experiences Widget Settings', 'destinely-experiences-widget'),
            __('Destinely Experiences Widget', 'destinely-experiences-widget'),
            'manage_options',
            'destinely-widget',
            array($this, 'settings_page')
        );
        
        // Enqueue admin assets
        add_action('load-' . $hook, array($this, 'enqueue_admin_assets'));
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Verify nonce on save
        if (isset($_POST['destinely_widget_nonce']) && !wp_verify_nonce($_POST['destinely_widget_nonce'], 'destinely_widget_settings')) {
            return;
        }
        
        register_setting('destinely_widget_settings', 'destinely_hotel_id', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('destinely_widget_settings', 'destinely_hotel_name', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('destinely_widget_settings', 'destinely_widget_enabled', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_auto_open', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_sound', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_notification', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_position', array('sanitize_callback' => array($this, 'sanitize_position')));
        register_setting('destinely_widget_settings', 'destinely_widget_trigger_delay', array('sanitize_callback' => 'absint'));
        register_setting('destinely_widget_settings', 'destinely_widget_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('destinely_widget_settings', 'destinely_widget_title', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('destinely_widget_settings', 'destinely_widget_section_title', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('destinely_widget_settings', 'destinely_widget_size', array('sanitize_callback' => array($this, 'sanitize_size')));
        register_setting('destinely_widget_settings', 'destinely_widget_show_prices', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_show_ratings', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_show_duration', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('destinely_widget_settings', 'destinely_widget_max_experiences', array('sanitize_callback' => 'absint'));
        register_setting('destinely_widget_settings', 'destinely_widget_status', array('sanitize_callback' => array($this, 'sanitize_status')));
    }

    /**
     * Sanitize checkbox value
     */
    public function sanitize_checkbox($value) {
        return $value ? 1 : 0;
    }

    /**
     * Sanitize position value
     */
    public function sanitize_position($value) {
        $allowed = array('bottom-right', 'bottom-left', 'top-right', 'top-left');
        return in_array($value, $allowed, true) ? $value : 'bottom-right';
    }

    /**
     * Sanitize size value
     */
    public function sanitize_size($value) {
        $allowed = array('small', 'medium', 'large');
        return in_array($value, $allowed, true) ? $value : 'medium';
    }

    /**
     * Sanitize status value
     */
    public function sanitize_status($value) {
        $allowed = array('active', 'inactive');
        return in_array($value, $allowed, true) ? $value : 'active';
    }

    /**
     * Settings page
     */
    public function settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'destinely-experiences-widget'));
        }
        ?>
        <div class="wrap destinely-admin-wrap">
            <div class="destinely-admin-header">
                <h1><?php echo esc_html__('Destinely Experiences Widget Settings', 'destinely-experiences-widget'); ?></h1>
                <p><?php echo esc_html__('Help your guests discover amazing local experiences and increase your revenue with our intelligent widget', 'destinely-experiences-widget'); ?></p>
            </div>
            
            <div class="destinely-status-card">
                <h3><?php echo esc_html__('Widget Status', 'destinely-experiences-widget'); ?></h3>
                <p><?php echo esc_html__('Configure your widget settings below to get started. The widget will fetch real experiences from our Supabase database.', 'destinely-experiences-widget'); ?></p>
            </div>
            
            <form method="post" action="options.php" id="destinely-widget-settings-form">
                <?php settings_fields('destinely_widget_settings'); ?>
                <?php wp_nonce_field('destinely_widget_settings', 'destinely_widget_nonce'); ?>
                <?php do_settings_sections('destinely_widget_settings'); ?>
                
                <table class="form-table destinely-form-table">
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_enabled"><?php echo esc_html__('Enable Widget', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_enabled" name="destinely_widget_enabled" value="1" <?php checked(get_option('destinely_widget_enabled'), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Enable the Destinely Experiences Widget on your website', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_hotel_id"><?php echo esc_html__('Hotel ID', 'destinely-experiences-widget'); ?> <span class="required">*</span></label></th>
                        <td>
                            <input type="text" id="destinely_hotel_id" name="destinely_hotel_id" value="<?php echo esc_attr(get_option('destinely_hotel_id')); ?>" class="regular-text" required />
                            <p class="description destinely-description"><?php echo esc_html__('Your unique hotel ID from Destinely dashboard', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_hotel_name"><?php echo esc_html__('Hotel Name', 'destinely-experiences-widget'); ?> <span class="required">*</span></label></th>
                        <td>
                            <input type="text" id="destinely_hotel_name" name="destinely_hotel_name" value="<?php echo esc_attr(get_option('destinely_hotel_name')); ?>" class="regular-text" required />
                            <p class="description destinely-description"><?php echo esc_html__('Your hotel name as it appears to guests', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_title"><?php echo esc_html__('Widget Title', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="text" id="destinely_widget_title" name="destinely_widget_title" value="<?php echo esc_attr(get_option('destinely_widget_title', __('Discover Local Experiences', 'destinely-experiences-widget'))); ?>" class="regular-text" />
                            <p class="description destinely-description"><?php echo esc_html__('Title displayed in the widget header', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_section_title"><?php echo esc_html__('Section Title', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="text" id="destinely_widget_section_title" name="destinely_widget_section_title" value="<?php echo esc_attr(get_option('destinely_widget_section_title', __('Nearby Experiences', 'destinely-experiences-widget'))); ?>" class="regular-text" />
                            <p class="description destinely-description"><?php echo esc_html__('Section heading above the experiences list (e.g. Nearby Experiences)', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_color"><?php echo esc_html__('Widget Color', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="color" id="destinely_widget_color" name="destinely_widget_color" value="<?php echo esc_attr(get_option('destinely_widget_color', '#2A9D8F')); ?>" />
                            <p class="description destinely-description"><?php echo esc_html__('Primary color for the widget (default: teal)', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_position"><?php echo esc_html__('Widget Position', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <select id="destinely_widget_position" name="destinely_widget_position">
                                <option value="bottom-right" <?php selected(get_option('destinely_widget_position', 'bottom-right'), 'bottom-right'); ?>><?php echo esc_html__('Bottom Right', 'destinely-experiences-widget'); ?></option>
                                <option value="bottom-left" <?php selected(get_option('destinely_widget_position', 'bottom-right'), 'bottom-left'); ?>><?php echo esc_html__('Bottom Left', 'destinely-experiences-widget'); ?></option>
                                <option value="top-right" <?php selected(get_option('destinely_widget_position', 'bottom-right'), 'top-right'); ?>><?php echo esc_html__('Top Right', 'destinely-experiences-widget'); ?></option>
                                <option value="top-left" <?php selected(get_option('destinely_widget_position', 'bottom-right'), 'top-left'); ?>><?php echo esc_html__('Top Left', 'destinely-experiences-widget'); ?></option>
                            </select>
                            <p class="description destinely-description"><?php echo esc_html__('Position of the widget on your website', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_auto_open"><?php echo esc_html__('Auto-Open Widget', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_auto_open" name="destinely_widget_auto_open" value="1" <?php checked(get_option('destinely_widget_auto_open', 0), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Automatically show widget after booking completion. Once opened, widget stays visible until user closes it.', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_trigger_delay"><?php echo esc_html__('Trigger Delay (seconds)', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="number" id="destinely_widget_trigger_delay" name="destinely_widget_trigger_delay" value="<?php echo esc_attr(get_option('destinely_widget_trigger_delay', 5)); ?>" min="0" max="60" step="1" />
                            <p class="description destinely-description"><?php echo esc_html__('Delay before widget appears after booking (0-60 seconds). Recommended: 5 seconds.', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_sound"><?php echo esc_html__('Notification Sound', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_sound" name="destinely_widget_sound" value="1" <?php checked(get_option('destinely_widget_sound'), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Play sound when widget appears', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_notification"><?php echo esc_html__('Show Notification Message', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_notification" name="destinely_widget_notification" value="1" <?php checked(get_option('destinely_widget_notification'), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Show notification message when widget appears', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_size"><?php echo esc_html__('Widget Size', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <select id="destinely_widget_size" name="destinely_widget_size">
                                <option value="small" <?php selected(get_option('destinely_widget_size', 'medium'), 'small'); ?>><?php echo esc_html__('Small', 'destinely-experiences-widget'); ?></option>
                                <option value="medium" <?php selected(get_option('destinely_widget_size', 'medium'), 'medium'); ?>><?php echo esc_html__('Medium', 'destinely-experiences-widget'); ?></option>
                                <option value="large" <?php selected(get_option('destinely_widget_size', 'medium'), 'large'); ?>><?php echo esc_html__('Large', 'destinely-experiences-widget'); ?></option>
                            </select>
                            <p class="description destinely-description"><?php echo esc_html__('Size of the widget container', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_max_experiences"><?php echo esc_html__('Max Experiences', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="number" id="destinely_widget_max_experiences" name="destinely_widget_max_experiences" value="<?php echo esc_attr(get_option('destinely_widget_max_experiences', 6)); ?>" min="1" max="20" step="1" />
                            <p class="description destinely-description"><?php echo esc_html__('Maximum number of experiences to display (1-20)', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_show_prices"><?php echo esc_html__('Show Prices', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_show_prices" name="destinely_widget_show_prices" value="1" <?php checked(get_option('destinely_widget_show_prices', 1), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Display experience prices in the widget', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_show_ratings"><?php echo esc_html__('Show Ratings', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_show_ratings" name="destinely_widget_show_ratings" value="1" <?php checked(get_option('destinely_widget_show_ratings', 1), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Display experience ratings in the widget', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_show_duration"><?php echo esc_html__('Show Duration', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <input type="checkbox" id="destinely_widget_show_duration" name="destinely_widget_show_duration" value="1" <?php checked(get_option('destinely_widget_show_duration', 1), 1); ?> />
                            <p class="description destinely-description"><?php echo esc_html__('Display experience duration in the widget', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><label for="destinely_widget_status"><?php echo esc_html__('Widget Status', 'destinely-experiences-widget'); ?></label></th>
                        <td>
                            <select id="destinely_widget_status" name="destinely_widget_status">
                                <option value="active" <?php selected(get_option('destinely_widget_status', 'active'), 'active'); ?>><?php echo esc_html__('Active', 'destinely-experiences-widget'); ?></option>
                                <option value="inactive" <?php selected(get_option('destinely_widget_status', 'active'), 'inactive'); ?>><?php echo esc_html__('Inactive', 'destinely-experiences-widget'); ?></option>
                            </select>
                            <p class="description destinely-description"><?php echo esc_html__('Enable or disable the widget', 'destinely-experiences-widget'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <div class="destinely-help-card">
                <h3><?php echo esc_html__('How it works:', 'destinely-experiences-widget'); ?></h3>
                <ol>
                    <li><?php echo esc_html__('Guests complete their hotel booking on your website', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('The Destinely widget automatically appears (if auto-open is enabled)', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Widget fetches real experiences from our Supabase database', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Guests can browse and book local experiences', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('You earn commission on every booking made through the widget', 'destinely-experiences-widget'); ?></li>
                </ol>
                
                <h3><?php echo esc_html__('Integration Methods:', 'destinely-experiences-widget'); ?></h3>
                <p><?php echo esc_html__('The widget automatically detects booking completion through:', 'destinely-experiences-widget'); ?></p>
                <ul>
                    <li><?php echo esc_html__('URL patterns (booking-confirmation, thank-you, etc.)', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('DOM elements with booking confirmation classes', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Custom JavaScript events from your booking system', 'destinely-experiences-widget'); ?></li>
                </ul>
                
                <h3><?php echo esc_html__('Features:', 'destinely-experiences-widget'); ?></h3>
                <ul>
                    <li><?php echo esc_html__('Real-time data from Supabase database', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Responsive design for all devices', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Image carousels for experiences', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Rating and pricing display', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Distance calculation from hotel', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Offline support and caching', 'destinely-experiences-widget'); ?></li>
                    <li><?php echo esc_html__('Analytics and tracking', 'destinely-experiences-widget'); ?></li>
                </ul>
                
                <h3><?php echo esc_html__('Need help?', 'destinely-experiences-widget'); ?></h3>
                <p>
                    <?php
                    printf(
                        /* translators: %1$s: Support URL, %2$s: Support email */
                        esc_html__('Visit our %1$s or contact us at %2$s', 'destinely-experiences-widget'),
                        '<a href="' . esc_url('https://www.destinely.com/support') . '" target="_blank" rel="noopener noreferrer">' . esc_html__('support center', 'destinely-experiences-widget') . '</a>',
                        '<a href="mailto:support@destinely.com">support@destinely.com</a>'
                    );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets() {
        wp_enqueue_script(
            'destinely-admin',
            DESTINELY_WIDGET_URL . 'assets/js/admin.js',
            array('jquery'),
            DESTINELY_WIDGET_VERSION,
            true
        );
        
        wp_localize_script('destinely-admin', 'destinely_admin', array(
            'assets_url' => DESTINELY_WIDGET_URL . 'assets',
            'site_url' => home_url(),
            'is_new_install' => !get_option('destinely_hotel_id')
        ));
    }

    /**
     * Enqueue scripts
     * 
     * Loads the main widget embed.js script which includes:
     * - Single slide-in animation from right/left (no double animations)
     * - Persistent widget that never disappears once opened
     * - Position locking to prevent resets
     * - Animation completion tracking to prevent re-animations
     */
    public function enqueue_scripts() {
        if (get_option('destinely_widget_enabled')) {
            $hotel_id = get_option('destinely_hotel_id');
            $hotel_name = get_option('destinely_hotel_name');
            
            if ($hotel_id && $hotel_name) {
                // Enqueue the main widget script with data attributes
                // This script includes all animation fixes and persistent widget behavior
                $api_base = apply_filters('destinely_api_base', 'https://www.destinely.com');
                wp_enqueue_script(
                    'destinely-widget',
                    esc_url($api_base . '/widget/embed.js'),
                    array(),
                    DESTINELY_WIDGET_VERSION,
                    true
                );
                
                // Add script tag with data attributes
                add_action('wp_head', array($this, 'add_widget_script_tag'));
            }
        }
    }

    /**
     * Add widget script tag with data attributes
     */
    public function add_widget_script_tag() {
        $hotel_id = get_option('destinely_hotel_id');
        $hotel_name = get_option('destinely_hotel_name');
        $auto_open = get_option('destinely_widget_auto_open');
        $sound = get_option('destinely_widget_sound');
        $notification = get_option('destinely_widget_notification');
        $position = get_option('destinely_widget_position', 'bottom-right');
        $color = get_option('destinely_widget_color', '#2A9D8F');
        $title = get_option('destinely_widget_title', __('Discover Local Experiences', 'destinely-experiences-widget'));
        $section_title = get_option('destinely_widget_section_title', __('Nearby Experiences', 'destinely-experiences-widget'));
        $trigger_delay = absint(get_option('destinely_widget_trigger_delay', 5)); // Default 5 seconds - matches widget behavior
        $size = get_option('destinely_widget_size', 'medium');
        $show_prices = get_option('destinely_widget_show_prices', 1);
        $show_ratings = get_option('destinely_widget_show_ratings', 1);
        $show_duration = get_option('destinely_widget_show_duration', 1);
        $max_experiences = absint(get_option('destinely_widget_max_experiences', 6));
        $status = get_option('destinely_widget_status', 'active');
        
        // Validate hotel_id - don't output widget if invalid
        if (empty($hotel_id) || $hotel_id === '00000000-0000-0000-0000-000000000000' || $hotel_id === '00000000-0000-0000-0000-000000000001') {
            // Invalid hotel ID - don't output widget
            return;
        }
        
        // Don't output widget if status is inactive
        if ($status === 'inactive') {
            return;
        }
        
        // API base URL - matches widget embed (animations, config API, track API)
        $api_base = apply_filters('destinely_api_base', 'https://www.destinely.com');
        
        // Build configuration array (matches embed.js exactly - all widget settings)
        $config = array(
            'hotelId' => $hotel_id,
            'hotelName' => $hotel_name,
            'autoOpen' => (bool) $auto_open,
            'notificationSound' => (bool) $sound,
            'notificationMessage' => (bool) $notification,
            'position' => $position,
            'color' => $color,
            'title' => $title,
            'section_title' => $section_title,
            'triggerDelay' => $trigger_delay,
            'size' => $size,
            'showPrices' => (bool) $show_prices,
            'showRatings' => (bool) $show_ratings,
            'showDuration' => (bool) $show_duration,
            'maxExperiences' => $max_experiences,
            'status' => $status,
            'apiBase' => $api_base,
            'assetBaseUrl' => $api_base  // Required for animations.js, animations.css
        );
        
        // Output script tag with properly escaped data attributes
        echo '<script type="text/javascript"';
        echo ' data-hotel-id="' . esc_attr($hotel_id) . '"';
        echo ' data-hotel-name="' . esc_attr($hotel_name) . '"';
        echo ' data-auto-open="' . esc_attr($auto_open ? 'true' : 'false') . '"';
        echo ' data-notification-sound="' . esc_attr($sound ? 'true' : 'false') . '"';
        echo ' data-notification-message="' . esc_attr($notification ? 'true' : 'false') . '"';
        echo ' data-position="' . esc_attr($position) . '"';
        echo ' data-color="' . esc_attr($color) . '"';
        echo ' data-title="' . esc_attr($title) . '"';
        echo ' data-section-title="' . esc_attr($section_title) . '"';
        echo ' data-trigger-delay="' . esc_attr($trigger_delay) . '"';
        echo ' data-size="' . esc_attr($size) . '"';
        echo ' data-show-prices="' . esc_attr($show_prices ? 'true' : 'false') . '"';
        echo ' data-show-ratings="' . esc_attr($show_ratings ? 'true' : 'false') . '"';
        echo ' data-show-duration="' . esc_attr($show_duration ? 'true' : 'false') . '"';
        echo ' data-max-experiences="' . esc_attr($max_experiences) . '"';
        echo ' data-status="' . esc_attr($status) . '">';
        echo "\n";
        echo '// Destinely Experiences Widget Configuration' . "\n";
        echo 'window.destinelyConfig = ' . wp_json_encode($config, JSON_UNESCAPED_SLASHES | JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) . ';' . "\n";
        echo '</script>' . "\n";
    }

    /**
     * Add widget to footer
     * 
     * Adds booking detection scripts that trigger the widget to appear.
     * The widget (loaded via embed.js) uses:
     * - Single slide-in animation (no double animations or disappearing)
     * - Persistent visibility (stays visible once opened)
     * - Position locking (prevents resets)
     */
    public function add_widget_to_footer() {
        if (get_option('destinely_widget_enabled')) {
            $hotel_id = get_option('destinely_hotel_id');
            $hotel_name = get_option('destinely_hotel_name');
            
            if ($hotel_id && $hotel_name) {
                ?>
                <script>
                    // Enhanced booking detection for WordPress sites
                    // Triggers widget to appear - widget handles its own animation once triggered
                    document.addEventListener('DOMContentLoaded', function() {
                        // Method 1: Listen for custom events from booking systems
                        document.addEventListener('bookingComplete', function(event) {
                            if (event.detail) {
                                const bookingEvent = new CustomEvent('hotelBookingComplete', {
                                    detail: {
                                        bookingComplete: true,
                                        guestEmail: event.detail.email || '',
                                        guestName: event.detail.name || '',
                                        bookingReference: event.detail.reference || ''
                                    }
                                });
                                window.dispatchEvent(bookingEvent);
                            }
                        });
                        
                        // Method 2: Auto-detect booking confirmation pages
                        const url = window.location.href.toLowerCase();
                        const bookingPatterns = [
                            'booking-confirmation', 'reservation-confirmed', 'booking-success',
                            'confirmation', 'thank-you', 'booking=confirmed', 'reservation=confirmed',
                            'payment-success', 'order-complete', 'booking-complete'
                        ];
                        
                        if (bookingPatterns.some(pattern => url.includes(pattern))) {
                            setTimeout(function() {
                                if (window.Destinely && window.Destinely.trigger) {
                                    window.Destinely.trigger('booking-confirmed');
                                }
                            }, 2000);
                        }
                        
                        // Method 3: Listen for common WordPress booking system events
                        const bookingSelectors = [
                            '.booking-confirmation', '.booking-success', '.confirmation-message',
                            '[data-booking-complete]', '#booking-confirmed', '.reservation-confirmed',
                            '.woocommerce-order-received', '.order-received', '.payment-success',
                            '.booking-complete', '.reservation-complete'
                        ];
                        
                        // Check if any booking confirmation elements exist
                        bookingSelectors.forEach(function(selector) {
                            const element = document.querySelector(selector);
                            if (element && element.offsetParent !== null) {
                                setTimeout(function() {
                                    if (window.Destinely && window.Destinely.trigger) {
                                        window.Destinely.trigger('booking-confirmed');
                                    }
                                }, 1500);
                            }
                        });
                        
                        // Method 4: WooCommerce specific detection
                        if (typeof wc_checkout_params !== 'undefined') {
                            // WooCommerce checkout completion
                            jQuery(document.body).on('checkout_complete', function() {
                                setTimeout(function() {
                                    if (window.Destinely && window.Destinely.trigger) {
                                        window.Destinely.trigger('booking-confirmed');
                                    }
                                }, 2000);
                            });
                        }
                        
                        // Method 5: Listen for URL changes (SPA support)
                        let currentUrl = window.location.href;
                        setInterval(function() {
                            if (window.location.href !== currentUrl) {
                                currentUrl = window.location.href;
                                if (bookingPatterns.some(pattern => currentUrl.toLowerCase().includes(pattern))) {
                                    setTimeout(function() {
                                        if (window.Destinely && window.Destinely.trigger) {
                                            window.Destinely.trigger('booking-confirmed');
                                        }
                                    }, 2000);
                                }
                            }
                        }, 1000);
                    });
                </script>
        <?php
            }
        }
    }
}

// Initialize the plugin
new Destinely_Widget();